import { City } from "../game/City.js";
import { UIManager } from "../ui/UIManager.js";
import { Drawable, Anchor } from "../ui/Drawable.js";
import { TextureInfo } from "../ui/TextureInfo.js";
import { IHasDrawable } from "../ui/IHasDrawable.js";
import { IOnResizeEvent } from "../ui/IOnResizeEvent.js";
import { StandardScroller } from "../ui/StandardScroller.js";
import { Resource, Flunds, Apples, Bread } from "../game/ResourceTypes.js";
import { TourismReward, ProductionReward } from "../game/EventTypes.js";
import { addResourceCosts, humanizeFloor } from "../ui/UIUtil.js";
import { inPlaceShuffle } from "../game/MiscFunctions.js";
import { OnePracticeRun, rangeMapLinear } from "./MinigameUtil.js";
import { v4 as uuidv4 } from "uuid"; // For unique order IDs

interface Order {
    id: string;
    name: string;
    requiredIngredients: string[]; // e.g., ["Bread", "Meat"]
    patience: number; // Seconds until customer leaves
    initialPatience: number; // For calculating speed bonuses
    bonusPoints: number;
}

interface Ingredient {
    type: string; // Matches ResourceTypes (e.g., "Bread")
    quantity: number;
    image: string; // Asset path (e.g., "minigame/truck/bread")
}

class TruckStopState {
    // Game state
    currentOrders: Order[] = [];
    preparedIngredients: string[] = []; // Ingredients in the "serve" zone
    ingredientStock: Ingredient[] = [];
    score = 0;
    timeRemaining = GAME_DURATION;
    gameStarted = false;
    userInputLocked = false;

    // UI state
    howToPlayShown = false;
    isPractice = false;
    selectedIngredient: string | null = null;
    endReason = "";
    winnings: Resource[] = [];

    // Timers (for cleanup)
    gameTimer: NodeJS.Timeout | null = null;
    patienceTimer: NodeJS.Timeout | null = null;
}

// Game constants (follow spec: magic numbers with units/comments)
const GAME_DURATION = 60; // Seconds per session
const MAX_ORDER_QUEUE = 5; // Max pending orders at once
const BASE_ORDER_SCORE = 10; // Points per basic correct order
const PATIENCE_BONUS_PER_SEC = 2; // Extra points per second saved on patience
const STARTING_INGREDIENTS: Ingredient[] = [
    { type: "Bread", quantity: 10, image: "minigame/truck/bread" },
    { type: "Meat", quantity: 8, image: "minigame/truck/meat" },
    { type: "Lettuce", quantity: 12, image: "minigame/truck/lettuce" },
    { type: "Cheese", quantity: 8, image: "minigame/truck/cheese" },
    { type: "Fries", quantity: 10, image: "minigame/truck/fries" },
];
const ORDER_DEFINITIONS = [
    { name: "Basic Burger", ingredients: ["Bread", "Meat"], patience: 20, bonus: 5 },
    { name: "Cheese Burger", ingredients: ["Bread", "Meat", "Cheese"], patience: 25, bonus: 15 },
    { name: "Veggie Wrap", ingredients: ["Lettuce", "Bread", "Cheese"], patience: 18, bonus: 10 },
    { name: "Fried Chicken", ingredients: ["Meat", "Fries"], patience: 22, bonus: 12 },
    { name: "Mega Meal", ingredients: ["Bread", "Meat", "Lettuce", "Cheese", "Fries"], patience: 30, bonus: 30 },
];

export class TruckStopHustle implements IHasDrawable, IOnResizeEvent {
    private state = new TruckStopState();
    private scroller = new StandardScroller(false, true); // Vertical scroll only
    private lastDrawable: Drawable | null = null;
    private preloaded = false;

    constructor(private city: City, private uiManager: UIManager) { }

    // ------------------------------
    // Core Game Lifecycle
    // ------------------------------
    public async preloadImages(): Promise<void> {
        if (this.preloaded) return;
        const assets = STARTING_INGREDIENTS.reduce((acc, ing) => ({ ...acc, [ing.image]: `assets/minigame/truck/${ing.type}.png` }), {});
        await this.uiManager.renderer.loadMoreSprites(this.city, assets);
        this.preloaded = true;
    }

    public startGame(): void {
        if (!this.city.checkAndSpendResources(this.getCosts())) return;

        this.initializeGame();
        this.state.gameStarted = true;
        this.state.userInputLocked = false;
        this.startGameTimers();
        this.uiManager.frameRequested = true;
    }

    private initializeGame(): void {
        // Reset state
        this.state.currentOrders = this.generateInitialOrders();
        this.state.ingredientStock = [...STARTING_INGREDIENTS];
        this.state.preparedIngredients = [];
        this.state.score = 0;
        this.state.timeRemaining = GAME_DURATION;
        this.state.endReason = "";

        // Clear timers (prevents leaks)
        if (this.state.gameTimer) clearTimeout(this.state.gameTimer);
        if (this.state.patienceTimer) clearTimeout(this.state.patienceTimer);
    }

    private startGameTimers(): void {
        // Main game timer (countdown)
        this.state.gameTimer = setTimeout(() => this.onGameTimeEnd(), GAME_DURATION * 1000);

        // Patience timer (update customer patience every second)
        this.state.patienceTimer = setInterval(() => this.updateCustomerPatience(), 1000);
    }

    private updateCustomerPatience(): void {
        if (!this.state.gameStarted) return;

        this.state.currentOrders = this.state.currentOrders.map(order => {
            const newPatience = order.patience - 1;
            if (newPatience <= 0) {
                this.state.score -= 5; // Penalty for lost customer
                return this.generateNewOrder(); // Replace with a new order
            }
            return { ...order, patience: newPatience };
        });

        this.uiManager.frameRequested = true;
    }

    private onGameTimeEnd(): void {
        this.endGame("Time's up!");
    }

    public endGame(reason: string): void {
        this.state.gameStarted = false
        this.state.userInputLocked = true;
        this.state.endReason = reason;

        // Clear timers
        if (this.state.gameTimer) clearTimeout(this.state.gameTimer);
        if (this.state.patienceTimer) clearInterval(this.state.patienceTimer);

        // Calculate rewards (skip for practice mode)
        if (!this.state.isPractice) {
            this.state.winnings = this.calculateRewards();
            this.city.transferResourcesFrom(this.state.winnings, "earn");
            this.city.events.push(new ProductionReward(1, rangeMapLinear(this.state.score, 0.1, 0.3, 0, 200, 0.01)));
        }

        this.uiManager.frameRequested = true;
    }

    // ------------------------------
    // Order & Ingredient Logic
    // ------------------------------
    private generateInitialOrders(): Order[] {
        const orders = [];
        const shuffledOrders = [...ORDER_DEFINITIONS].sort(() => Math.random() - 0.5);
        for (let i = 0; i < MAX_ORDER_QUEUE; i++) {
            orders.push(this.createOrder(shuffledOrders[i]));
        }
        return orders;
    }

    private generateNewOrder(): Order {
        const randomOrder = ORDER_DEFINITIONS[Math.floor(Math.random() * ORDER_DEFINITIONS.length)];
        return this.createOrder(randomOrder);
    }

    private createOrder(def: typeof ORDER_DEFINITIONS[number]): Order {
        return {
            id: uuidv4(),
            name: def.name,
            requiredIngredients: [...def.ingredients],
            patience: def.patience,
            initialPatience: def.patience,
            bonusPoints: def.bonus,
        };
    }

    public handleIngredientClick(ingredientType: string): void {
        if (this.state.userInputLocked || !this.state.gameStarted) return;

        // Deselect if clicking the same ingredient again
        if (this.state.selectedIngredient === ingredientType) {
            this.state.selectedIngredient = null;
            this.state.preparedIngredients = this.state.preparedIngredients.filter(i => i !== ingredientType);
            return;
        }

        // Select new ingredient (if stock allows)
        const stock = this.state.ingredientStock.find(i => i.type === ingredientType);
        if (stock && stock.quantity > 0) {
            this.state.selectedIngredient = ingredientType;
            this.state.preparedIngredients.push(ingredientType);
        }

        this.uiManager.frameRequested = true;
    }

    public handleServeClick(): void {
        if (this.state.userInputLocked || !this.state.gameStarted || !this.state.currentOrders.length) return;

        const currentOrder = this.state.currentOrders[0];
        const isMatch = this.isOrderFulfilled(currentOrder, this.state.preparedIngredients);

        if (isMatch) {
            this.state.score += this.calculateOrderScore(currentOrder);
            this.state.currentOrders.shift(); // Remove first order
            this.state.currentOrders.push(this.generateNewOrder()); // Add new order to queue
            this.deductUsedIngredients(currentOrder.requiredIngredients);
        } else {
            this.state.score -= 3; // Penalty for wrong order
        }

        // Reset preparation zone
        this.state.preparedIngredients = [];
        this.state.selectedIngredient = null;
        this.uiManager.frameRequested = true;
    }

    private isOrderFulfilled(order: Order, prepared: string[]): boolean {
        // Check if prepared ingredients exactly match the order (order doesn't matter)
        const sortedPrepared = [...prepared].sort();
        const sortedRequired = [...order.requiredIngredients].sort();
        return JSON.stringify(sortedPrepared) === JSON.stringify(sortedRequired);
    }

    private calculateOrderScore(order: Order): number {
        const timeSaved = order.initialPatience - order.patience;
        return BASE_ORDER_SCORE + order.bonusPoints + (timeSaved * PATIENCE_BONUS_PER_SEC);
    }

    private deductUsedIngredients(ingredients: string[]): void {
        ingredients.forEach(type => {
            const stock = this.state.ingredientStock.find(i => i.type === type);
            if (stock) stock.quantity--;
        });
    }

    // ------------------------------
    // Rewards & Costs
    // ------------------------------
    public getCosts(): { type: string; amount: number }[] {
        return this.state.isPractice ? OnePracticeRun : [{ type: "Flunds", amount: 500 }];
    }

    private calculateRewards(): Resource[] {
        const score = this.state.score;
        const flunds = rangeMapLinear(score, 100, 500, 0, 200, 1);
        const apples = rangeMapLinear(score, 2, 5, 0, 100, 1);
        const bread = score > 150 ? 1 : 0;

        return [
            new Flunds(flunds),
            new Apples(apples),
            ...(bread > 0 ? [new Bread(bread)] : []),
        ].filter(r => r.amount > 0);
    }

    // ------------------------------
    // UI Rendering (IHasDrawable)
    // ------------------------------
    public asDrawable(): Drawable {
        if (!this.state.shown) return this.lastDrawable = new Drawable({ width: "0px" });

        const main = new Drawable({
            width: "100%",
            height: "100%",
            fallbackColor: "#1a1a1a",
        });

        if (!this.state.gameStarted) {
            this.drawStartOverlay(main);
        } else {
            this.drawGameArea(main);
        }

        return this.lastDrawable = main;
    }

    private drawStartOverlay(parent: Drawable): void {
        const overlay = parent.addChild(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            width: "min(100%, 500px)",
            height: "100%",
            fallbackColor: "#2d2d2d",
            onDrag: (_, y) => this.scroller.handleDrag(y, overlay.screenArea),
            onDragEnd: () => this.scroller.resetDrag(),
        }));

        if (this.state.howToPlayShown) {
            this.drawHowToPlay(overlay);
            return;
        }

        let nextY = 10 - this.scroller.getScroll();
        // Add title, start button, practice toggle, how-to-play, and winnings display
        this.drawStartButton(overlay, nextY);
        nextY += 100;
        this.drawPracticeToggle(overlay, nextY);
        nextY += 60;
        this.drawHowToPlayButton(overlay, nextY);
        nextY += 60;
        this.drawWinnings(overlay, nextY);

        this.scroller.setChildrenSize(nextY - 10);
    }

    private drawGameArea(parent: Drawable): void {
        // Timer bar (top)
        this.drawTimer(parent);

        // Order queue (left)
        const orderQueue = parent.addChild(new Drawable({
            x: 20,
            y: 60,
            width: "40%",
            height: "calc(100% - 80px)",
            fallbackColor: "#333333",
        }));
        this.drawOrders(orderQueue);

        // Ingredient station (right)
        const ingredientStation = parent.addChild(new Drawable({
            x: "60%",
            y: 60,
            width: "35%",
            height: "calc(100% - 80px)",
            fallbackColor: "#333333",
        }));
        this.drawIngredients(ingredientStation);

        // Preparation zone (bottom center)
        this.drawPreparationZone(parent);
    }

    private drawTimer(parent: Drawable): void {
        parent.addChild(new Drawable({
            anchors: ["centerX"],
            y: 10,
            width: "90%",
            height: "30px",
            children: [
                new Drawable({
                    width: "100%",
                    height: "100%",
                    fallbackColor: "#555555",
                    image: new TextureInfo(200, 20, "ui/progressbg"),
                }),
                new Drawable({
                    width: "100%",
                    height: "100%",
                    clipWidth: this.state.timeRemaining / GAME_DURATION,
                    fallbackColor: "#4CAF50",
                    image: new TextureInfo(200, 20, "ui/progressfg"),
                }),
                new Drawable({
                    anchors: ["centerX"],
                    y: 4,
                    text: `${humanizeFloor(this.state.timeRemaining)}s`,
                    color: "white",
                }),
            ],
        }));
    }

    private drawOrders(parent: Drawable): void {
        let nextY = 10;
        this.state.currentOrders.forEach(order => {
            const orderItem = parent.addChild(new Drawable({
                x: 10,
                y: nextY,
                width: "calc(100% - 20px)",
                height: "60px",
                fallbackColor: "#444444",
                children: [
                    new Drawable({
                        x: 10,
                        y: 10,
                        width: "80%",
                        text: order.name,
                        color: "white",
                    }),
                    new Drawable({
                        x: 10,
                        y: 35,
                        width: "100%",
                        height: "15px",
                        children: [
                            new Drawable({
                                width: "100%",
                                height: "100%",
                                fallbackColor: "#777777",
                            }),
                            new Drawable({
                                width: `${(order.patience / order.initialPatience) * 100}%`,
                                height: "100%",
                                fallbackColor: order.patience < 5 ? "#FF5252" : "#4CAF50",
                            }),
                        ],
                    }),
                    new Drawable({
                        x: "90%",
                        y: 38,
                        text: `${order.patience}s`,
                        color: "white",
                        rightAlign: true,
                    }),
                ],
            }));
            nextY += 70;
        });
    }

    private drawIngredients(parent: Drawable): void {
        let nextX = 10;
        let nextY = 10;
        const ingPerRow = 3;

        this.state.ingredientStock.forEach(ing => {
            const ingButton = parent.addChild(new Drawable({
                x: nextX,
                y: nextY,
                width: "50px",
                height: "50px",
                image: new TextureInfo(64, 64, ing.image),
                fallbackColor: ing.quantity === 0 ? "#555555" : "#333333",
                onClick: () => this.handleIngredientClick(ing.type),
                grayscale: ing.quantity === 0,
                children: [
                    new Drawable({
                        anchors: ["bottomRight"],
                        x: -5,
                        y: -5,
                        width: "20px",
                        height: "20px",
                        text: ing.quantity.toString(),
                        color: "white",
                        rightAlign: true,
                    }),
                ],
            }));

            nextX += 60;
            if (nextX > parent.widthPx! - 60) {
                nextX = 10;
                nextY += 60;
            }
        });
    }

    private drawPreparationZone(parent: Drawable): void {
        const zone = parent.addChild(new Drawable({
            anchors: ["centerX"],
            y: "85%",
            width: "80%",
            height: "100px",
            fallbackColor: "#444444",
            children: [
                new Drawable({
                    anchors: ["centerX"],
                    y: 10,
                    text: "Prepare Order",
                    color: "white",
                }),
                ...this.state.preparedIngredients.map(ing => {
                    const stock = this.state.ingredientStock.find(i => i.type === ing);
                    return new Drawable({
                        x: 10 + (this.state.preparedIngredients.indexOf(ing) * 60),
                        y: 40,
                        width: "50px",
                        height: "50px",
                        image: new TextureInfo(64, 64, stock?.image || ""),
                    });
                }),
            ],
        }));

        // Serve button
        zone.addChild(new Drawable({
            anchors: ["centerX"],
            y: 70,
            width: "120px",
            height: "40px",
            fallbackColor: "#4CAF50",
            onClick: () => this.handleServeClick(),
            children: [
                new Drawable({
                    anchors: ["centerX"],
                    y: 8,
                    text: "Serve",
                    color: "white",
                }),
            ],
        }));
    }

    // ------------------------------
    // UI Helpers
    // ------------------------------
    private drawStartButton(parent: Drawable, y: number): void {
        const button = parent.addChild(new Drawable({
            anchors: ["centerX"],
            y,
            width: "220px",
            height: "50px",
            fallbackColor: "#4CAF50",
            onClick: () => this.startGame(),
            children: [
                new Drawable({
                    anchors: ["centerX"],
                    y: 12,
                    text: "Start Hustle",
                    color: "white",
                }),
            ],
        }));
        addResourceCosts(button, this.getCosts(), 80, 15, false, false, false, 48, 10);
    }

    private drawPracticeToggle(parent: Drawable, y: number): void {
        parent.addChild(new Drawable({
            anchors: ["centerX"],
            y,
            width: "80%",
            height: "50px",
            fallbackColor: "#333333",
            onClick: () => this.state.isPractice = !this.state.isPractice,
            children: [
                new Drawable({
                    x: 10,
                    width: "40px",
                    height: "40px",
                    image: new TextureInfo(64, 64, this.state.isPractice ? "ui/checked" : "ui/unchecked"),
                }),
                new Drawable({
                    x: 60,
                    y: 12,
                    text: this.state.isPractice ? "Practice Mode (No Rewards)" : "Play for Rewards",
                    color: "white",
                }),
            ],
        }));
    }

    private drawHowToPlay(overlay: Drawable): void {
        overlay.addChild(new Drawable({
            anchors: ["centerX"],
            y: 20,
            text: "Truck Stop Hustle Rules",
            color: "white",
        }));

        const rules = overlay.addChild(new Drawable({
            x: 20,
            y: 60,
            width: "calc(100% - 40px)",
            height: "400px",
            wordWrap: true,
            keepParentWidth: true,
            text: `
        - Fulfill customer orders before their patience runs out.
        - Click ingredients to add them to your preparation zone.
        - Click "Serve" when your order matches the first customer's request.
        - Earn points for speed and complex orderslose points for mistakes!
        - Game ends after 60 seconds; rewards based on score.
      `,
            color: "white",
        }));

        this.scroller.setChildrenSize(500);
    }

    private drawHowToPlayButton(parent: Drawable, y: number): void {
        parent.addChild(new Drawable({
            anchors: ["centerX"],
            y,
            width: "220px",
            height: "50px",
            fallbackColor: "#2196F3",
            onClick: () => this.state.howToPlayShown = !this.state.howToPlayShown,
            children: [
                new Drawable({
                    anchors: ["centerX"],
                    y: 12,
                    text: "How to Play",
                    color: "white",
                }),
            ],
        }));
    }

    private drawWinnings(parent: Drawable, y: number): void {
        if (!this.state.winnings.length) return;

        const winningsArea = parent.addChild(new Drawable({
            anchors: ["X"],
            y,
            width: "80%",
            height: "150px",
            fallbackColor: "#333333",
        }));

        winningsArea.addChild(new Drawable({
            anchors: ["centerX"],
            y: 10,
            text: "Last Winnings",
            color: "white",
        }));

        const costDrawable = winningsArea.addChild(new Drawable({
            x: 10,
            y: 40,
            width: "calc(100% - 20px)",
            height: "100px",
        }));
        addResourceCosts(costDrawable, this.state.winnings, 0, 0, false, false, false, 48, 10);
    }

    // ------------------------------
    // Resize Handling (IOnResizeEvent)
    // ------------------------------
    public onResize(): void {
        this.scroller.onResize();
        this.uiManager.frameRequested = true;
    }

    public getLastDrawable(): Drawable | null {
        return this.lastDrawable;
    }
}
