import { TitleTypes } from "../game/AchievementTypes.js";
import { City } from "../game/City.js";
import { CityFlags } from "../game/CityFlags.js";
import { ProductionReward } from "../game/EventTypes.js";
import { LONG_TICKS_PER_DAY } from "../game/FundamentalConstants.js";
import { UIManager } from "../ui/UIManager.js";
import { Resource } from "../game/Resource.js";
import { Flunds, Research, getResourceType } from "../game/ResourceTypes.js";
import { TextureInfo } from "../ui/TextureInfo.js";
import { Drawable } from "../ui/Drawable.js";
import { addResourceCosts, humanizeFloor } from "../ui/UIUtil.js";
import { IHasDrawable } from "../ui/IHasDrawable.js";
import { IOnResizeEvent } from "../ui/IOnResizeEvent.js";
import { StandardScroller } from "../ui/StandardScroller.js";
import { drawMinigameOptions } from "../ui/MinigameOptions.js";
import { OnePracticeRun, progressMinigameOptionResearch, filterConvertAwardWinnings, rangeMapLinear } from "./MinigameUtil.js";

// === CONSTANTS ===

const FLOOR_WIDTH = 10;
const FLOOR_HEIGHT = 15;
const TILE_SIZE = 48;
const GAME_DURATION = 120; // seconds

enum PlantType {
    SpiderPlant = "spiderplant",
    SnakePlant = "snakeplant",
    PeaceLily = "peacelily",
    AloeVera = "aloevera",
    Bamboo = "bamboo",
}

interface PlantDef {
    id: PlantType;
    name: string;
    color: string;
    pollutionTypes: ('co2' | 'particulates' | 'nox')[];
    growthTime: number; // seconds per stage
    waterCost: number;
    pollutionRemoval: number; // per tick when mature
    harvestReward: Resource[];
    icon: string;
}

const PLANT_TYPES: Record<PlantType, PlantDef> = {
    [PlantType.SpiderPlant]: {
        id: PlantType.SpiderPlant,
        name: "Spider Plant",
        color: "#4CAF50",
        pollutionTypes: ['co2'],
        growthTime: 1,
        waterCost: 1,
        pollutionRemoval: 0.8,
        harvestReward: [new Flunds(8)],
        icon: "minigame/greenthumb/spiderplant",
    },
    [PlantType.SnakePlant]: {
        id: PlantType.SnakePlant,
        name: "Snake Plant",
        color: "#2E7D32",
        pollutionTypes: ['particulates'],
        growthTime: 2,
        waterCost: 2,
        pollutionRemoval: 1.5,
        harvestReward: [new Flunds(12), new Research(1)],
        icon: "minigame/greenthumb/snakeplant",
    },
    [PlantType.PeaceLily]: {
        id: PlantType.PeaceLily,
        name: "Peace Lily",
        color: "#8BC34A",
        pollutionTypes: ['nox'],
        growthTime: 3,
        waterCost: 3,
        pollutionRemoval: 2.2,
        harvestReward: [new Flunds(15), new Research(2)],
        icon: "minigame/greenthumb/peacelily",
    },
    [PlantType.AloeVera]: {
        id: PlantType.AloeVera,
        name: "Aloe Vera",
        color: "#689F38",
        pollutionTypes: ['co2', 'particulates', 'nox'],
        growthTime: 3,
        waterCost: 2,
        pollutionRemoval: 0.6,
        harvestReward: [new Flunds(5), new Resource("VitaminB12", 1)],
        icon: "minigame/greenthumb/aloevera",
    },
    [PlantType.Bamboo]: {
        id: PlantType.Bamboo,
        name: "Bamboo",
        color: "#33691E",
        pollutionTypes: ['co2', 'particulates'],
        growthTime: 1,
        waterCost: 2,
        pollutionRemoval: 1.2,
        harvestReward: [new Flunds(10), new Resource("Paper", 1)],
        icon: "minigame/greenthumb/bamboo",
    },
};

type Difficulty = 'easy' | 'medium' | 'hard';

interface DifficultySettings {
    waterStart: number;
    pollutionRate: number;
    rewardMultiplier: number;
}

const DIFFICULTY_SETTINGS: Record<Difficulty, DifficultySettings> = {
    easy: { waterStart: 15, pollutionRate: 0.2, rewardMultiplier: 1 },
    medium: { waterStart: 10, pollutionRate: 0.3, rewardMultiplier: 1.25 },
    hard: { waterStart: 7, pollutionRate: 0.5, rewardMultiplier: 1.75 },
};

interface PlantTile {
    plantType: PlantType | null;
    stage: 0 | 1 | 2 | 3; // 0=empty, 1=seed, 2=sprout, 3=mature
    harvestable?: boolean;
}

interface GreenThumbState {
    pollution: number; // 0100
    timer: number;     // remaining seconds
    garden: PlantTile[][]; // 10x15 grid
    water: number;     // available water
    selectedPlant: PlantType | null;
    difficulty: Difficulty;
    growthTimeouts: { x: number, y: number, timeout: NodeJS.Timeout }[];
    lastWaterCost: number;
}

// === MAIN CLASS ===

export class GreenThumbMinigame implements IHasDrawable, IOnResizeEvent {
    private state: GreenThumbState;
    private uiManager: UIManager;
    private city: City;
    private scroller: StandardScroller;

    // UI State
    public shown: boolean = false;
    public gameStarted: boolean = false;
    public isPractice: boolean = false;
    public howToPlayShown: boolean = false;
    public userInputLocked: boolean = false;
    public endReason: string = "";
    public winnings: Resource[] | null = null;
    public selectedDifficulty: Difficulty = 'medium';

    // Timers
    private timerTimeout: NodeJS.Timeout | null = null;
    private growthTimeouts: { x: number, y: number, timeout: NodeJS.Timeout }[] = [];

    // Drawable caching
    private lastDrawable: Drawable | null = null;
    private preloaded: boolean = false;

    constructor(city: City, uiManager: UIManager) {
        this.city = city;
        this.uiManager = uiManager;
        this.scroller = new StandardScroller(false, true); // vertical only
        this.state = {
            pollution: 30,
            timer: GAME_DURATION,
            garden: Array(FLOOR_HEIGHT).fill(null).map(() => Array(FLOOR_WIDTH).fill(null).map(() => ({ plantType: null, stage: 0 }))),
            water: DIFFICULTY_SETTINGS[this.selectedDifficulty].waterStart,
            selectedPlant: null,
            difficulty: this.selectedDifficulty,
            growthTimeouts: [],
            lastWaterCost: 0,
        };

        this.selectedPlant = Object.values(PLANT_TYPES)[0].id;
    }

    // === PUBLIC INTERFACE ===

    public show(): void {
        this.shown = true;
        this.preloadImages();
        this.uiManager.frameRequested = true;
    }

    public hide(): void {
        this.shown = false;
        this.gameStarted = false;
        this.clearTimers();
        this.uiManager.frameRequested = true;
    }

    public startGame(): void {
        if (!this.city.checkAndSpendResources(this.getCosts())) return;

        this.initializeGame();
        this.city.updateLastUserActionTime();
        this.game.fullSave();
        this.gameStarted = true;
        this.startTimer();
        this.uiManager.frameRequested = true;
    }

    public endGame(): void {
        this.clearTimers();
        this.gameStarted = false;
        this.winnings = this.calculateWinnings();
        this.uiManager.frameRequested = true;
    }

    public onResize(): void {
        this.scroller.onResize();
        this.uiManager.frameRequested = true;
    }

    public asDrawable(): Drawable {
        if (!this.shown) return this.lastDrawable = new Drawable({ width: "0px" });

        const mainDrawable = new Drawable({
            width: "100%",
            height: "100%",
            fallbackColor: '#222222',
        });

        if (!this.gameStarted) {
            this.drawStartOverlay(mainDrawable);
            if (!this.howToPlayShown) this.drawCloseButton(mainDrawable);
        } else {
            this.drawGameArea(mainDrawable);
        }

        this.lastDrawable = mainDrawable;
        return mainDrawable;
    }

    public getLastDrawable(): Drawable | null {
        return this.lastDrawable;
    }

    // === GAME LOGIC ===

    private initializeGame(): void {
        this.state = {
            pollution: 30,
            timer: GAME_DURATION,
            garden: Array(FLOOR_HEIGHT).fill(null).map(() => Array(FLOOR_WIDTH).fill(null).map(() => ({ plantType: null, stage: 0 }))),
            water: DIFFICULTY_SETTINGS[this.selectedDifficulty].waterStart,
            selectedPlant: Object.values(PLANT_TYPES)[0].id,
            difficulty: this.selectedDifficulty,
            growthTimeouts: [],
            lastWaterCost: 0,
        };
        this.clearTimers();
        this.winnings = null;
        this.endReason = "";
    }

    private getCosts(): { type: string, amount: number }[] {
        return this.isPractice ? OnePracticeRun : [
            { type: "flunds", amount: 500 },
            { type: "research", amount: 1 }
        ];
    }

    private calculateWinnings(): Resource[] {
        const baseReward = 200 + rangeMapLinear(this.state.pollution, 0, 200, 100, 0, 1); // Inverse: lower pollution = higher reward
        const flunds = new Flunds(baseReward * DIFFICULTY_SETTINGS[this.state.difficulty].rewardMultiplier);
        const research = new Research(Math.floor(this.state.pollution < 20 ? 3 : 1));

        // Apply city-wide effect: reduce particulate pollution for 1 day
        const effect = new (require("../game/GridType").Effect)(
            EffectType.ParticulatePollution,
            -0.3, // reduces pollution by 30%
            undefined,
            undefined,
            LONG_TICKS_PER_DAY
        );
        this.city.spreadEffect(effect, 3, 3, true);

        // Award bonus event
        this.city.events.push(new ProductionReward(12, 0.1));

        // Unlock achievement if pollution < 10%
        if (this.state.pollution < 10) {
            this.city.checkAndAwardAchievement("GreenThumbMaster");
        }

        return filterConvertAwardWinnings(this.city, [flunds, research]);
    }

    private startTimer(): void {
        this.clearTimers();
        this.timerTimeout = setTimeout(() => {
            if (!this.gameStarted) return;

            // City emits pollution
            this.state.pollution = Math.min(100, this.state.pollution + DIFFICULTY_SETTINGS[this.state.difficulty].pollutionRate);

            // Apply plant effects
            this.applyPlantEffects();

            // Decrease timer
            this.state.timer--;

            if (this.state.timer <= 0 || this.state.pollution >= 100) {
                this.endReason = this.state.pollution >= 100 ? "Air is too toxic!" : "Time's up!";
                this.endGame();
            } else {
                this.startTimer();
            }

            this.uiManager.frameRequested = true;
        }, 1000);
    }

    private applyPlantEffects(): void {
        for (let y = 0; y < FLOOR_HEIGHT; y++) {
            for (let x = 0; x < FLOOR_WIDTH; x++) {
                const tile = this.state.garden[y]?.[x];
                if (!tile || tile.stage < 3) continue; // Only mature plants

                const plant = PLANT_TYPES[tile.plantType!];
                for (const pType of plant.pollutionTypes) {
                    if (pType === 'co2') this.state.pollution -= plant.pollutionRemoval * 0.4;
                    if (pType === 'particulates') this.state.pollution -= plant.pollutionRemoval * 0.35;
                    if (pType === 'nox') this.state.pollution -= plant.pollutionRemoval * 0.25;
                }
            }
        }
        this.state.pollution = Math.max(0, this.state.pollution);
    }

    private clearTimers(): void {
        if (this.timerTimeout) clearTimeout(this.timerTimeout);
        this.timerTimeout = null;

        for (const t of this.growthTimeouts) {
            clearTimeout(t.timeout);
        }
        this.growthTimeouts = [];
    }

    private selectPlant(plantId: PlantType): void {
        if (!this.gameStarted || this.userInputLocked) return;
        this.selectedPlant = plantId;
        this.state.lastWaterCost = PLANT_TYPES[plantId].waterCost;
        this.uiManager.frameRequested = true;
    }

    private handleTileClick(x: number, y: number): void {
        if (!this.gameStarted || this.userInputLocked || !this.selectedPlant) return;

        const plantDef = PLANT_TYPES[this.selectedPlant];
        if (this.state.water < plantDef.waterCost) return;

        const tile = this.state.garden[y][x];
        if (tile.plantType !== null) return; // Already planted

        // Plant!
        tile.plantType = this.selectedPlant;
        tile.stage = 1;
        this.state.water -= plantDef.waterCost;

        // Start growth timer
        const timeout = setTimeout(() => {
            if (!this.gameStarted) return;
            if (tile.stage >= 3) return;

            tile.stage++;
            if (tile.stage === 3) tile.harvestable = true;

            this.uiManager.frameRequested = true;
        }, plantDef.growthTime * 1000);

        this.growthTimeouts.push({ x, y, timeout });
        this.uiManager.frameRequested = true;
    }

    private toggleRules(): void {
        this.howToPlayShown = !this.howToPlayShown;
        if (this.howToPlayShown) {
            this.scroller.resetScroll();
        }
        this.uiManager.frameRequested = true;
    }

    // === UI DRAWING METHODS ===

    private drawStartOverlay(parent: Drawable): void {
        const overlay = parent.addChild(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            width: "min(100%, 600px)",
            height: "100%",
            fallbackColor: '#111111',
            id: "startOverlay",
            onDrag: (x: number, y: number) => { this.scroller.handleDrag(y, overlay.screenArea); },
            onDragEnd: () => { this.scroller.resetDrag(); },
        }));

        if (this.howToPlayShown) {
            this.drawHowToPlay(overlay);
            return;
        }

        let nextY = 10 - this.scroller.getScroll();
        let baseY = nextY;

        // Title
        overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "100%",
            height: "48px",
            text: "Green Thumb",
        }));
        nextY += 134;

        // Start Button
        const startButton = overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "220px",
            height: "48px",
            fallbackColor: '#444444',
            onClick: () => this.startGame(),
            children: [
                new Drawable({
                    anchors: ["centerX"],
                    y: 5,
                    width: "calc(100% - 10px)",
                    height: "100%",
                    text: "Start Game",
                    centerOnOwnX: true
                })
            ]
        }));

        const unaffordable = !this.city.hasResources(this.getCosts(), false);
        addResourceCosts(startButton, this.getCosts(), 86 - (this.isPractice ? 0 : 0), 58, false, false, false, 48, 10, 32, undefined, undefined, unaffordable, this.city);
        nextY += 176;

        // Practice Toggle
        overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "500px",
            height: "48px",
            fallbackColor: '#00000000',
            onClick: () => { this.isPractice = !this.isPractice; },
            children: [
                new Drawable({
                    x: 5,
                    width: "48px",
                    height: "48px",
                    image: new TextureInfo(64, 64, this.isPractice ? "ui/checked" : "ui/unchecked"),
                }),
                new Drawable({
                    anchors: ["right"],
                    rightAlign: true,
                    x: 5,
                    y: 7,
                    width: "calc(100% - 60px)",
                    height: "100%",
                    text: "Practice Run (no rewards)",
                }),
            ]
        }));
        nextY += 60;

        // How to Play Button
        overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "220px",
            height: "48px",
            fallbackColor: '#444444',
            onClick: () => { this.toggleRules(); },
            children: [
                new Drawable({
                    anchors: ["centerX"],
                    y: 5,
                    width: "calc(100% - 10px)",
                    height: "100%",
                    text: "How to Play",
                    centerOnOwnX: true
                })
            ]
        }));
        nextY += 60;

        // Difficulty Selector
        overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "500px",
            height: "48px",
            fallbackColor: '#00000000',
            text: `Difficulty: ${this.selectedDifficulty.charAt(0).toUpperCase() + this.selectedDifficulty.slice(1)}`,
        }));
        nextY += 20;

        const difficultyOptions: { text: string, value: Difficulty }[] = [
            { text: "Easy", value: 'easy' },
            { text: "Medium", value: 'medium' },
            { text: "Hard", value: 'hard' },
        ];

        for (let i = 0; i < difficultyOptions.length; i++) {
            const opt = difficultyOptions[i];
            const isSelected = this.selectedDifficulty === opt.value;
            const btn = overlay.addChild(new Drawable({
                anchors: ['centerX'],
                centerOnOwnX: true,
                y: nextY,
                width: "150px",
                height: "36px",
                fallbackColor: isSelected ? '#00ff11' : '#333333',
                onClick: () => { this.selectedDifficulty = opt.value; },
                children: [
                    new Drawable({
                        anchors: ["centerX"],
                        centerOnOwnX: true,
                        y: 5,
                        width: "100%",
                        height: "100%",
                        text: opt.text,
                        centerOnOwnX: true,
                        grayscale: !isSelected,
                    })
                ]
            }));
            nextY += 40;
        }

        // Winnings from last play
        if (this.winnings?.length) {
            const winningsArea = overlay.addChild(new Drawable({
                anchors: ['centerX'],
                centerOnOwnX: true,
                y: nextY,
                width: "min(100%, 500px)",
                height: "400px",
                fallbackColor: '#444444',
                id: "winningsArea"
            }));

            winningsArea.addChild(new Drawable({
                anchors: ['centerX'],
                centerOnOwnX: true,
                biggerOnMobile: true,
                scaleYOnMobile: true,
                y: 10,
                width: "250px",
                height: "32px",
                text: "Pollution reduced: " + Math.round(100 - this.state.pollution) + "%",
            }));

            winningsArea.addChild(new Drawable({
                anchors: ['centerX'],
                centerOnOwnX: true,
                biggerOnMobile: true,
                scaleYOnMobile: true,
                y: 58,
                width: "250px",
                height: "32px",
                text: "Rewards earned:",
            }));

            const rewardContainer = winningsArea.addChild(new Drawable({
                x: 107,
                y: 100,
                width: "100%",
                fallbackColor: '#00000000',
                scaleYOnMobile: true
            }));

            addResourceCosts(rewardContainer, this.winnings, 0, 0, false, false, false, 64, 10, 32, 4);

            nextY += 420;
        }

        // Minigame Options
        nextY = drawMinigameOptions(this.city, overlay, nextY, [
            { group: "gt-r", id: "0", text: "Hydroponics (+water)", icon: "resource/water" },
            { group: "gt-r", id: "1", text: "Airflow Boost (+removal)", icon: "resource/air" }
        ]);

        this.scroller.setChildrenSize(nextY - baseY);
    }

    private drawHowToPlay(parent: Drawable): void {
        let nextY = 10 - this.scroller.getScroll();

        parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "100%",
            height: "48px",
            text: "Green Thumb Rules",
        }));
        nextY += 60;

        const rules = [
            "Tap a plant in the selector to choose it.",
            "Tap an empty tile in the garden to plant it.",
            "Each plant needs water  you start with " + DIFFICULTY_SETTINGS[this.selectedDifficulty].waterStart + " units.",
            "Plants grow slowly  wait for them to mature (stage 3).",
            "Mature plants clean the air  more plants = cleaner city.",
            "Pollution rises slowly. If it hits 100%, you lose.",
            "Use practice mode to learn without spending resources.",
            "Research upgrades unlock powerful plant traits."
        ];

        for (const rule of rules) {
            parent.addChild(new Drawable({
                x: 20,
                y: nextY,
                width: "calc(100% - 40px)",
                height: "40px",
                wordWrap: true,
                keepParentWidth: true,
                text: rule,
            }));
            nextY += 50;
        }

        this.scroller.setChildrenSize(nextY - 10);
    }

    private drawGameArea(parent: Drawable): void {
        let nextY = 0;

        // Pollution meter and timer
        nextY = this.drawPollutionBar(parent, nextY);
        nextY = this.drawTimer(parent, nextY);
        nextY = this.drawWaterCounter(parent, nextY);

        // Garden grid
        nextY = this.drawGarden(parent, nextY);

        // Plant selector
        nextY = this.drawSelectorMenu(parent, nextY);

        // Give up button
        if (this.gameStarted) {
            const giveUpBtn = parent.addChild(new Drawable({
                anchors: ['centerX'],
                centerOnOwnX: true,
                y: nextY,
                width: "200px",
                height: "48px",
                fallbackColor: '#8B0000',
                onClick: () => this.endGame(),
                children: [
                    new Drawable({
                        anchors: ["centerX"],
                        y: 5,
                        width: "calc(100% - 10px)",
                        height: "100%",
                        text: "Give Up",
                        centerOnOwnX: true
                    })
                ]
            }));
        }
    }

    private drawPollutionBar(parent: Drawable, nextY: number): number {
        const width = FLOOR_WIDTH * TILE_SIZE;
        parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: width + "px",
            height: "30px",
            fallbackColor: '#111111',
            image: new TextureInfo(200, 20, "ui/progressbg"),
            children: [
                new Drawable({
                    clipWidth: 1 - (this.state.pollution / 100),
                    width: "100%",
                    height: "100%",
                    noXStretch: false,
                    fallbackColor: '#00ff11',
                    image: new TextureInfo(200, 20, "ui/progressfg"),
                    reddize: this.state.pollution > 80,
                })
            ]
        }));

        parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY + 32,
            width: "100%",
            height: "24px",
            text: `Pollution: ${Math.round(this.state.pollution)}%`,
            centerOnOwnX: true,
            grayscale: this.state.pollution > 80,
        }));

        return nextY + 60;
    }

    private drawTimer(parent: Drawable, nextY: number): number {
        const width = FLOOR_WIDTH * TILE_SIZE;
        parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: width + "px",
            height: "30px",
            fallbackColor: '#111111',
            image: new TextureInfo(200, 20, "ui/progressbg"),
            children: [
                new Drawable({
                    clipWidth: this.state.timer / GAME_DURATION,
                    width: "100%",
                    height: "100%",
                    noXStretch: false,
                    fallbackColor: '#0099ff',
                    image: new TextureInfo(200, 20, "ui/progressfg"),
                })
            ]
        }));

        parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY + 32,
            width: "100%",
            height: "24px",
            text: `Time: ${this.state.timer}s`,
            centerOnOwnX: true,
        }));

        return nextY + 60;
    }

    private drawWaterCounter(parent: Drawable, nextY: number): number {
        const width = FLOOR_WIDTH * TILE_SIZE;
        parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: width + "px",
            height: "30px",
            fallbackColor: '#111111',
            image: new TextureInfo(200, 20, "ui/progressbg"),
            children: [
                new Drawable({
                    clipWidth: this.state.water / 15, // Max water = 15
                    width: "100%",
                    height: "100%",
                    noXStretch: false,
                    fallbackColor: '#00aaff',
                    image: new TextureInfo(200, 20, "ui/progressfg"),
                })
            ]
        }));

        parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY + 32,
            width: "100%",
            height: "24px",
            text: `Water: ${this.state.water}`,
            centerOnOwnX: true,
        }));

        return nextY + 60;
    }

    private drawGarden(parent: Drawable, nextY: number): number {
        const tileWidth = TILE_SIZE;
        const tileHeight = TILE_SIZE;

        const baseY = nextY;

        for (let y = 0; y < FLOOR_HEIGHT; y++) {
            for (let x = 0; x < FLOOR_WIDTH; x++) {
                const tile = this.state.garden[y]?.[x];
                if (!tile) continue;

                const plantDef = PLANT_TYPES[tile.plantType!];
                const stage = tile.stage;

                const xPix = x * tileWidth;
                const yPix = y * tileHeight;

                const img = `minigame/greenthumb/${plantDef.id}${stage === 0 ? '' : stage === 1 ? 'seed' : stage === 2 ? 'sprout' : 'mature'}`;

                parent.addChild(new Drawable({
                    x: xPix,
                    y: yPix + nextY,
                    width: tileWidth + "px",
                    height: tileHeight + "px",
                    image: new TextureInfo(TILE_SIZE, TILE_SIZE, img),
                    onClick: () => this.handleTileClick(x, y),
                    fallbackColor: '#333333',
                    reddize: stage === 3 && tile.harvestable,
                }));
            }
        }

        return nextY + FLOOR_HEIGHT * tileHeight;
    }

    private drawSelectorMenu(parent: Drawable, nextY: number): number {
        const selectorArea = parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "100%",
            height: TILE_SIZE + 20 + "px",
            fallbackColor: '#111111'
        }));

        const plantTypes = Object.values(PLANT_TYPES);
        for (let i = 0; i < plantTypes.length; i++) {
            const p = plantTypes[i];
            const isSelected = this.selectedPlant === p.id;

            const btn = selectorArea.addChild(new Drawable({
                x: i * (TILE_SIZE + 12),
                y: 0,
                width: TILE_SIZE + "px",
                height: TILE_SIZE + "px",
                image: new TextureInfo(TILE_SIZE, TILE_SIZE, p.icon),
                onClick: () => this.selectPlant(p.id),
                fallbackColor: isSelected ? '#00ff11' : '#444444',
                grayscale: !this.state.water >= p.waterCost,
                reddize: !this.state.water >= p.waterCost,
            }));

            // Show cost badge
            if (p.waterCost > 0) {
                btn.addChild(new Drawable({
                    anchors: ['bottom', 'right'],
                    x: -4,
                    y: -4,
                    width: "20px",
                    height: "20px",
                    fallbackColor: '#00000088',
                    text: p.waterCost.toString(),
                    centerOnOwnX: true,
                    centerOnOwnY: true,
                    fontSize: 14,
                    grayscale: true
                }));
            }
        }

        return nextY + TILE_SIZE + 30;
    }

    private drawCloseButton(parent: Drawable): void {
        parent.addChild(new Drawable({
            anchors: ['top', 'right'],
            x: -10,
            y: 10,
            width: "40px",
            height: "40px",
            image: new TextureInfo(40, 40, "ui/close"),
            onClick: () => this.hide(),
            fallbackColor: '#555555'
        }));
    }

    public async preloadImages(): Promise<void> {
        if (this.preloaded) return;

        const urls: { [key: string]: string } = {};

        for (const plant of Object.values(PLANT_TYPES)) {
            urls[`minigame/greenthumb/${plant.id}`] = `assets/minigame/greenthumb/${plant.id}.png`;
            urls[`minigame/greenthumb/${plant.id}seed`] = `assets/minigame/greenthumb/${plant.id}seed.png`;
            urls[`minigame/greenthumb/${plant.id}sprout`] = `assets/minigame/greenthumb/${plant.id}sprout.png`;
            urls[`minigame/greenthumb/${plant.id}mature`] = `assets/minigame/greenthumb/${plant.id}mature.png`;
        }

        await this.uiManager.renderer.loadMoreSprites(this.city, urls);
        this.preloaded = true;
    }
}
