//Qwen3-Coder-30B-A3B didn't include the imports.
import { City } from "../game/City.js";
import { Resource } from "../game/Resource.js";
import { Drawable } from "../ui/Drawable.js";
import { IHasDrawable } from "../ui/IHasDrawable.js";
import { IOnResizeEvent } from "../ui/IOnResizeEvent.js";
import { StandardScroller } from "../ui/StandardScroller.js";
import { TextureInfo } from "../ui/TextureInfo.js";
import { UIManager } from "../ui/UIManager.js";
import { addResourceCosts } from "../ui/UIUtil.js";

interface ARPState {
    grid: (string | null)[][]; // Grid representation with resource types
    playerPosition: { x: number; y: number }; // Current player position
    resources: Resource[]; // Current carried resources
    energy: number; // Current energy level
    level: number; // Current progression level
    timeLeft: number; // Time remaining in seconds
    score: number; // Current score
    collectedShrines: number; // Count of shrines visited this level
    maxShrines: number; // Total number of shrines per level
    goals: Goal[]; // List of goals to complete this level
    destinationZones: DestinationZone[]; // Locations where items must be delivered
    sourceShrines: SourceShrine[]; // Locations where resources originate
    currentMode: 'collect' | 'deliver'; // Player mode

    // Visual state properties
    playerDirection: 'up' | 'down' | 'left' | 'right';
    drawOffset: number; // For animations
}

interface Goal {
    type: string; // e.g., barley, coal, electronics
    amount: number;
    conveyed: boolean;
}

interface DestinationZone {
    x: number;
    y: number;
    type: string;
    requiredAmount: number;
    deliveredAmount: number;
    requiredTypes: string[];
}

interface SourceShrine {
    x: number;
    y: number;
    type: string;
    availableAmount: number;
    capacityLimit: number;
}

export class AdvancedRelocationProtocol implements IHasDrawable, IOnResizeEvent {
    private state: ARPState;
    private uiManager: UIManager;
    private timerTimeout: NodeJS.Timeout | null = null;
    private animateTimeout: NodeJS.Timeout | null = null;
    private lastDrawable: Drawable | null = null;
    private preloaded: boolean = false;
    private scroller: StandardScroller = new StandardScroller(false, true);
    private maxEnergy: number = 100;
    private timeLimitBase: number = 60;

    // Constants for level progression
    private static readonly GRID_WIDTH = 8;
    private static readonly GRID_HEIGHT = 8;
    private static readonly ENERGY_PER_MOVE = 5;
    private static readonly LEVEL_COSTS = [100, 150, 200, 250, 300];
    private static readonly TIME_BONUS_MULTIPLIER = 0.8;

    // Gameplay constants
    private static readonly MOVE_DIAGONAL = true;
    private static readonly SOURCE_SHRINE_TYPES = ['barley', 'coal', 'electronics', 'diamonds'];
    private static readonly DESTINATION_ZONE_TYPES = ['storage', 'market', 'factory', 'warehouse'];
    private static readonly PLAYER_SPRITE = 'minigame/arp_player';
    private static readonly DEFAULT_GRID_SYMBOLS = {
        'empty': 'minigame/arp_empty',
        'wall': 'minigame/arp_wall',
        'shrine': 'minigame/arp_shrine',
        'destination': 'minigame/arp_destination'
    }

    constructor(private city: City, private uiManager: UIManager) {
        this.state = this.initializeState();
    }

    private initializeState(): ARPState {
        return {
            grid: [],
            playerPosition: { x: 0, y: 0 },
            resources: [],
            energy: 100,
            level: 1,
            timeLeft: 0,
            score: 0,
            collectedShrines: 0,
            maxShrines: 0,
            goals: [],
            destinationZones: [],
            sourceShrines: [],
            currentMode: 'collect',
            playerDirection: 'right',
            drawOffset: 0
        };
    }

    // Core gameplay methods
    private startNewLevel(): void {
        this.state.timeLeft = this.timeLimitBase + this.state.level * 10;
        this.state.energy = this.maxEnergy;
        this.state.resources = [];
        this.state.collectedShrines = 0;
        this.generateLevel();
        this.startTimer();
        this.uiManager.frameRequested = true;
    }

    private generateLevel(): void {
        // Reset grid
        this.state.grid = Array(AdvancedRelocationProtocol.GRID_HEIGHT).fill(null).map(() =>
            Array(AdvancedRelocationProtocol.GRID_WIDTH).fill(null)
        );

        // Set player start position
        this.state.playerPosition = { x: 4, y: 0 };

        // Place walls and obstacles
        this.placeGenericWalls();

        // Generate shrines and destinations
        this.generateSourceShrines();
        this.generateDestinationZones();

        // Define goals based on level
        this.generateGoals();
    }

    private placeGenericWalls(): void {
        // Simple wall patterns for easier levels
        const pattern = [[2, 2], [4, 2], [2, 4], [4, 4]];
        for (const [x, y] of pattern) {
            this.state.grid[y][x] = 'wall';
        }
    }

    private generateSourceShrines(): void {
        this.state.sourceShrines = [];
        const shrineTypes = AdvancedRelocationProtocol.SOURCE_SHRINE_TYPES;
        const numShrines = Math.min(3 + this.state.level, 6);

        for (let i = 0; i < numShrines; i++) {
            let x, y;
            do {
                x = Math.floor(Math.random() * AdvancedRelocationProtocol.GRID_WIDTH);
                y = Math.floor(Math.random() * AdvancedRelocationProtocol.GRID_HEIGHT);
            } while (this.state.grid[y][x] !== null ||
                (x === this.state.playerPosition.x && y === this.state.playerPosition.y));

            this.state.grid[y][x] = 'shrine';
            this.state.sourceShrines.push({
                x: x,
                y: y,
                type: shrineTypes[Math.floor(Math.random() * shrineTypes.length)],
                availableAmount: Math.max(3, 5 - this.state.level) + Math.floor(Math.random() * 3),
                capacityLimit: 5
            });
        }
    }

    private generateDestinationZones(): void {
        this.state.destinationZones = [];
        const zoneTypes = AdvancedRelocationProtocol.DESTINATION_ZONE_TYPES;
        const numZones = Math.min(2 + Math.floor(this.state.level / 2), 4);

        for (let i = 0; i < numZones; i++) {
            let x, y;
            do {
                x = Math.floor(Math.random() * AdvancedRelocationProtocol.GRID_WIDTH);
                y = Math.floor(Math.random() * AdvancedRelocationProtocol.GRID_HEIGHT);
            } while (this.state.grid[y][x] !== null ||
                (x === this.state.playerPosition.x && y === this.state.playerPosition.y));

            this.state.grid[y][x] = 'destination';
            this.state.destinationZones.push({
                x: x,
                y: y,
                type: zoneTypes[i % zoneTypes.length],
                requiredAmount: 3 + this.state.level + Math.floor(Math.random() * 2),
                deliveredAmount: 0,
                requiredTypes: [this.state.sourceShrines[i % this.state.sourceShrines.length].type]
            });
        }
    }

    private generateGoals(): void {
        this.state.goals = [];
        const resources = this.state.sourceShrines.map(s => s.type);

        // Create a mix of easy and hard goal requirements
        for (let i = 0; i < 2 + Math.floor(this.state.level / 3); i++) {
            const type = resources[Math.floor(Math.random() * resources.length)];
            const amount = Math.max(1, Math.floor(Math.random() * Math.ceil(this.state.level / 2)) + 1);

            this.state.goals.push({
                type: type,
                amount: amount,
                conveyed: false
            });
        }
    }

    private startTimer(): void {
        if (this.timerTimeout) clearTimeout(this.timerTimeout);
        this.timerTimeout = setTimeout(() => {
            if (!this.gameStarted) return;
            this.state.timeLeft--;
            if (this.state.timeLeft <= 0) {
                this.endReason = "Time's up!";
                this.endGame();
            } else {
                this.startTimer();
            }
            this.uiManager.frameRequested = true;
        }, 1000);
    }

    // Movement and interaction methods
    public movePlayer(dx: number, dy: number): void {
        if (this.state.energy < AdvancedRelocationProtocol.ENERGY_PER_MOVE) return;
        if (this.state.drawOffset > 0) return;

        const newX = this.state.playerPosition.x + dx;
        const newY = this.state.playerPosition.y + dy;

        // Check bounds
        if (newX < 0 || newX >= AdvancedRelocationProtocol.GRID_WIDTH ||
            newY < 0 || newY >= AdvancedRelocationProtocol.GRID_HEIGHT) {
            return;
        }

        // Check if the tile is valid to move into
        const gridValue = this.state.grid[newY][newX];
        if (gridValue === 'wall') return;

        // Update player direction for animation
        this.state.playerDirection =
            dx > 0 ? 'right' : dx < 0 ? 'left' :
                dy > 0 ? 'down' : 'up';

        // Handle resource collection
        if (gridValue === 'shrine') {
            const shrine = this.state.sourceShrines.find(s => s.x === newX && s.y === newY);
            if (shrine && shrine.availableAmount > 0) {
                this.collectResource(shrine);
            }
        }

        // Handle delivery to zones
        if (gridValue === 'destination') {
            this.deliverToZone(newX, newY);
        }

        // Move player and deduce energy
        this.state.playerPosition = { x: newX, y: newY };
        this.state.energy -= AdvancedRelocationProtocol.ENERGY_PER_MOVE;

        // Check if we've reached a condition to end level
        if (this.checkLevelCompletion()) {
            this.completeLevel();
        }
    }

    private collectResource(shrine: SourceShrine): void {
        if (shrine.availableAmount <= 0) return;

        // Add resource to inventory
        const resourceExists = this.state.resources.some(r => r.type === shrine.type);
        if (!resourceExists) {
            this.state.resources.push(new Resource(shrine.type, 1));
        } else {
            this.state.resources = this.state.resources.map(r =>
                r.type === shrine.type ? new Resource(r.type, r.amount + 1) : r
            );
        }

        // Deplete shrine
        shrine.availableAmount--;
        this.state.collectedShrines++;

        // Check if this was the only shrine we need to visit
        if (this.state.collectedShrines >= this.state.sourceShrines.length) {
            // Possibly activate slow-time effect here
            this.uiManager.frameRequested = true;
        } else {
            this.uiManager.frameRequested = true;
        }
    }

    private deliverToZone(x: number, y: number): void {
        const zone = this.state.destinationZones.find(z => z.x === x && z.y === y);
        if (!zone) return;

        // Try to deliver resources
        const resource = this.state.resources.find(r => r.type === zone.requiredTypes[0]);
        if (resource && resource.amount > 0) {
            zone.deliveredAmount++;

            // Remove resource from inventory
            this.state.resources = this.state.resources.filter(r => r.type !== zone.requiredTypes[0]);

            // If all delivered
            if (zone.deliveredAmount >= zone.requiredAmount) {
                zone.deliveredAmount = zone.requiredAmount;
                // Mark as completed? Maybe trigger reward or bonus?

                // Update goals if this completes any
                this.state.goals = this.state.goals.map(g =>
                    g.type === zone.requiredTypes[0] && g.amount <= zone.requiredAmount ? { ...g, conveyed: true } : g
                );
            }

            this.uiManager.frameRequested = true;
        }
    }

    private checkLevelCompletion(): boolean {
        // Check if all goals are met
        const allGoalsConveyed = this.state.goals.every(g => g.conveyed);
        return allGoalsConveyed;
    }

    private completeLevel(): void {
        // Add score for level complete
        const levelBonus = Math.floor(50 * (this.state.level + 1));
        const timeBonus = Math.floor(this.state.timeLeft * 0.5);
        const energyBonus = Math.floor(this.state.energy / 2);

        this.state.score += levelBonus + timeBonus + energyBonus;
        this.state.level++;

        // Begin next level after delay
        setTimeout(() => {
            this.uiManager.frameRequested = true;
        }, 1000);
    }

    // Utility functions
    public isAffordable(cost: number): boolean {
        return this.state.resources.some(r => r.type === 'flunds' && r.amount >= cost);
    }

    public getCosts(): { type: string, amount: number }[] {
        // Each level costs a fixed amount of flunds
        return [{ type: 'flunds', amount: AdvancedRelocationProtocol.LEVEL_COSTS[Math.min(this.state.level - 1, 4)] }];
    }

    // UI and drawing methods
    public asDrawable(): Drawable {
        if (!this.shown) return this.lastDrawable = new Drawable({ width: "0px" });

        const mainDrawable = new Drawable({
            width: "100%",
            height: "100%",
            fallbackColor: '#222222',
        });

        if (!this.gameStarted) {
            this.drawStartOverlay(mainDrawable);
        } else {
            this.drawGameArea(mainDrawable);
        }

        this.lastDrawable = mainDrawable;
        return mainDrawable;
    }

    private drawGameArea(parent: Drawable): void {
        let nextY = this.drawGrid(parent, 80);
        nextY = this.drawProgressBar(parent, nextY);
        nextY = this.drawTimer(parent, nextY);
        nextY = this.drawResources(parent, nextY);
        nextY = this.drawStatusPanel(parent, nextY);
    }

    private drawGrid(parent: Drawable, startY: number): number {
        const GRID_SIZE = 50;
        const START_X = (window.innerWidth - (AdvancedRelocationProtocol.GRID_WIDTH * GRID_SIZE)) / 2;
        const START_Y = startY;

        // Draw the entire grid
        for (let y = 0; y < AdvancedRelocationProtocol.GRID_HEIGHT; y++) {
            for (let x = 0; x < AdvancedRelocationProtocol.GRID_WIDTH; x++) {
                const cellX = START_X + (x * GRID_SIZE);
                const cellY = START_Y + (y * GRID_SIZE);

                let backgroundImage = AdvancedRelocationProtocol.DEFAULT_GRID_SYMBOLS.empty;
                let clickHandler = null;

                const gridValue = this.state.grid[y][x];
                if (gridValue === 'wall') {
                    backgroundImage = AdvancedRelocationProtocol.DEFAULT_GRID_SYMBOLS.wall;
                } else if (gridValue === 'shrine') {
                    backgroundImage = AdvancedRelocationProtocol.DEFAULT_GRID_SYMBOLS.shrine;
                    clickHandler = () => {
                        const shrine = this.state.sourceShrines.find(s => s.x === x && s.y === y);
                        if (shrine) this.handleShrineClick(shrine);
                    };
                } else if (gridValue === 'destination') {
                    backgroundImage = AdvancedRelocationProtocol.DEFAULT_GRID_SYMBOLS.destination;
                    clickHandler = () => {
                        const zone = this.state.destinationZones.find(z => z.x === x && z.y === y);
                        if (zone) this.handleDeliveryClick(zone);
                    };
                }

                // Draw tile
                const drawable = parent.addChild(new Drawable({
                    x: cellX,
                    y: cellY,
                    width: GRID_SIZE + 'px',
                    height: GRID_SIZE + 'px',
                    image: new TextureInfo(GRID_SIZE, GRID_SIZE, backgroundImage),
                    onClick: clickHandler
                }));

                // Highlight if it's player location
                if (x === this.state.playerPosition.x && y === this.state.playerPosition.y) {
                    drawable.children.push(new Drawable({
                        width: '100%',
                        height: '100%',
                        image: new TextureInfo(GRID_SIZE, GRID_SIZE, AdvancedRelocationProtocol.PLAYER_SPRITE),
                        centerOnOwnX: true,
                        y: 0
                    }));
                }
            }
        }

        return START_Y + (AdvancedRelocationProtocol.GRID_HEIGHT * GRID_SIZE) + 20;
    }

    private handleShrineClick(shrine: SourceShrine): void {
        if (this.state.energy < AdvancedRelocationProtocol.ENERGY_PER_MOVE) return;
        if (shrine.availableAmount <= 0) {
            this.uiManager.frameRequested = true;
            return;
        }

        // Make a small clink sound or visual effect here if we want
        this.movePlayer(shrine.x - this.state.playerPosition.x, shrine.y - this.state.playerPosition.y);
    }

    private handleDeliveryClick(zone: DestinationZone): void {
        if (this.state.energy < AdvancedRelocationProtocol.ENERGY_PER_MOVE) return;

        // Make sure we're carrying appropriate resources
        if (this.state.resources.some(x => x.type === zone.requiredTypes[0])) {
            this.movePlayer(zone.x - this.state.playerPosition.x, zone.y - this.state.playerPosition.y);
        }
    }

    private drawProgressBar(parent: Drawable, nextY: number): number {
        const BAR_WIDTH = 300;
        const BAR_HEIGHT = 25;
        const progress = (this.state.timeLeft / this.timeLimitBase) * 100;

        parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: BAR_WIDTH + "px",
            height: BAR_HEIGHT + "px",
            fallbackColor: '#666666',
            image: new TextureInfo(BAR_WIDTH, BAR_HEIGHT, "ui/progressbg"),
            children: [
                new Drawable({
                    clipWidth: progress,
                    width: "100%",
                    height: "100%",
                    noXStretch: false,
                    fallbackColor: '#ff9900',
                    image: new TextureInfo(BAR_WIDTH, BAR_HEIGHT, "ui/progressfg"),
                })
            ]
        }));
        nextY += 10 + BAR_HEIGHT;

        return nextY;
    }

    private drawTimer(parent: Drawable, nextY: number): number {
        parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "160px",
            height: "30px",
            text: `Time Left: ${this.state.timeLeft}`,
            centerOnOwnX: true,
            biggerOnMobile: true
        }));
        nextY += 40;
        return nextY;
    }

    private drawResources(parent: Drawable, nextY: number): number {
        parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "100%",
            height: "20px",
            text: this.state.resources.map(r => r.type + ":" + r.amount).join(', '),
            centerOnOwnX: true,
            biggerOnMobile: true
        }));
        nextY += 30;
        return nextY;
    }

    private drawStatusPanel(parent: Drawable, nextY: number): number {
        parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "300px",
            height: "100px",
            text: `Level: ${this.state.level}  Energy: ${this.state.energy}/100  Goals: ${this.state.goals.filter(g => g.conveyed).length}/${this.state.goals.length}`,
            centerOnOwnX: true,
            biggerOnMobile: true
        }));
        nextY += 110;
        return nextY;
    }

    // UI overlay components
    private drawStartOverlay(parent: Drawable): void {
        const overlay = parent.addChild(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            width: "min(100%, 600px)",
            height: "100%",
            fallbackColor: '#111111',
            id: "startOverlay",
            onDrag: (x: number, y: number) => { this.scroller.handleDrag(y, overlay.screenArea); },
            onDragEnd: () => { this.scroller.resetDrag(); },
        }));

        if (this.howToPlayShown) {
            this.drawHowToPlay(overlay, parent);
            return;
        }

        let nextY = 10 - this.scroller.getScroll();
        let baseY = nextY;

        overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "100%",
            height: "48px",
            text: "Advanced Relocation Protocol",
            biggerOnMobile: true
        }));
        nextY += 134;

        const startButton = overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "220px",
            height: "48px",
            fallbackColor: '#444444',
            onClick: () => this.startGame(),
            children: [
                new Drawable({
                    anchors: ["centerX"],
                    y: 5,
                    width: "calc(100% - 10px)",
                    height: "100%",
                    text: "Start Game",
                    centerOnOwnX: true
                })
            ]
        }));

        const unaffordable = !this.city.hasResources(this.getCosts(), false);
        addResourceCosts(startButton, this.getCosts(), 86, 58, false, false, false, 48, 10, 32, undefined, undefined, unaffordable, this.city);
        nextY += 176;

        overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "500px",
            height: "48px",
            fallbackColor: '#00000000',
            onClick: () => { this.isPractice = !this.isPractice; },
            children: [
                new Drawable({
                    x: 5,
                    width: "48px",
                    height: "48px",
                    image: new TextureInfo(64, 64, this.isPractice ? "ui/checked" : "ui/unchecked"),
                }),
                new Drawable({
                    anchors: ["right"],
                    rightAlign: true,
                    x: 5,
                    y: 7,
                    width: "calc(100% - 60px)",
                    height: "100%",
                    text: "Practice Run (no rewards)",
                }),
            ]
        }));
        nextY += 60;

        // How to play button
        overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "220px",
            height: "48px",
            fallbackColor: '#444444',
            onClick: () => { this.toggleRules(); },
            children: [
                new Drawable({
                    anchors: ["centerX"],
                    y: 5,
                    width: "calc(100% - 10px)",
                    height: "100%",
                    text: "How to Play",
                    centerOnOwnX: true
                })
            ]
        }));
        nextY += 60;

        if (this.winnings?.length) {
            // Draw the winnings from the last playthrough
            const winningsArea = overlay.addChild(new Drawable({
                anchors: ['centerX'],
                centerOnOwnX: true,
                y: nextY,
                width: "min(100%, 500px)",
                height: "500px",
                fallbackColor: '#444444',
                id: "winningsArea"
            }));

            winningsArea.addChild(new Drawable({
                anchors: ['centerX'],
                centerOnOwnX: true,
                biggerOnMobile: true,
                scaleYOnMobile: true,
                y: 10,
                width: "250px",
                height: "32px",
                text: "Levels completed: " + (this.state.level - 1),
            }));

            winningsArea.addChild(new Drawable({
                anchors: ['centerX'],
                centerOnOwnX: true,
                biggerOnMobile: true,
                scaleYOnMobile: true,
                y: 58,
                width: "250px",
                height: "32px",
                text: "Rewards attained:",
            }));
            winningsArea.addChild(new Drawable({
                x: 107,
                y: 100,
                width: "100%",
                fallbackColor: '#00000000',
                scaleYOnMobile: true
            }));
            addResourceCosts(winningsArea.children[winningsArea.children.length - 1], this.winnings, 0, 0, false, false, false, 64, 10, 32, 4);
            nextY += 510;
        }

        this.scroller.setChildrenSize(nextY - baseY);
    }

    private drawHowToPlay(overlay: Drawable, root: Drawable): void {
        let parent = overlay;
        parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: 10 - this.scroller.getScroll(),
            width: "100%",
            height: "48px",
            text: "ARP How to Play",
            biggerOnMobile: true
        }));

        root.onClick = () => this.toggleRules();

        parent = parent.addChild(new Drawable({
            x: 20,
            y: 80 - this.scroller.getScroll(),
            width: "calc(100% - 40px)",
            height: "40px",
            wordWrap: true,
            keepParentWidth: true,
            text: "Transport resources from shrines to destination zones before time runs out!",
        }));

        parent = parent.addChild(new Drawable({
            anchors: ['bottom'],
            y: -50,
            width: "calc(100% - 40px)",
            height: "40px",
            wordWrap: true,
            keepParentWidth: true,
            text: "Winning is based on how quickly you complete the level and how advanced it is.",
        }));

        parent = parent.addChild(new Drawable({
            x: 20,
            y: 150,
            width: "calc(100% - 40px)",
            height: "100%",
            wordWrap: true,
            keepParentWidth: true,
            text: "Controls: Use arrow buttons or swipe to move. Eat resources when you're close. " +
                "Point at a resource shrine to indicate what to collect, then move to it. " +
                "Always move carefully with your energy!",
        }));

        this.scroller.setChildrenSize(600);
    }

    // Lifecycle methods
    public async preloadImages(): Promise<void> {
        if (this.preloaded) return;
        const urls: { [key: string]: string } = {
            "minigame/arp_player": "assets/minigame/arp_player.png",
            "minigame/arp_empty": "assets/minigame/arp_empty.png",
            "minigame/arp_wall": "assets/minigame/arp_wall.png",
            "minigame/arp_shrine": "assets/minigame/arp_shrine.png",
            "minigame/arp_destination": "assets/minigame/arp_destination.png",
        };

        this.uiManager.renderer.loadMoreSprites(this.city, urls);
        this.preloaded = true;
    }

    public show(): void {
        this.shown = true;
        if (this.preloaded) {
            // Load assets if needed
        }
    }

    public hide(): void {
        this.shown = false;
    }

    public toggleRules(): void {
        this.howToPlayShown = !this.howToPlayShown;
        if (this.howToPlayShown) {
            this.scroller.resetScroll();
        }
    }

    public startGame(): void {
        if (this.city.checkAndSpendResources(this.getCosts())) {
            this.initializeGame();
            this.city.updateLastUserActionTime();
            this.game.fullSave();
        }
    }

    public initializeGame(): void {
        this.state = this.initializeState();
        this.gameStarted = true;
        this.state.level = 1;
        this.startNewLevel();
        this.uiManager.frameRequested = true;
    }

    // ...rest of the methods go here based on provided template
}
