import { TitleTypes } from "../game/AchievementTypes.js";
import { City } from "../game/City.js";
import { CityFlags } from "../game/CityFlags.js";
import { ProductionReward, TourismReward } from "../game/EventTypes.js";
import { LONG_TICKS_PER_DAY } from "../game/FundamentalConstants.js";
import { UIManager } from "../ui/UIManager.js";
import { Resource } from "../game/Resource.js";
import { Flunds, getResourceType } from "../game/ResourceTypes.js";
import { TextureInfo } from "../ui/TextureInfo.js";
import { Drawable } from "../ui/Drawable.js";
import { addResourceCosts, humanizeFloor, longTicksToDaysAndHours } from "../ui/UIUtil.js";
import { IHasDrawable } from "../ui/IHasDrawable.js";
import { IOnResizeEvent } from "../ui/IOnResizeEvent.js";
import { StandardScroller } from "../ui/StandardScroller.js";
import { inPlaceShuffle } from "../game/MiscFunctions.js";
import { GameState } from "../game/GameState.js";
import { drawMinigameOptions } from "../ui/MinigameOptions.js";
import { OnePracticeRun, progressMinigameOptionResearch, rangeMapLinear } from "./MinigameUtil.js";
import { EffectType } from "../game/GridType.js";

const GRID_SIZE = 6;
const GAME_DURATION = 60; // seconds
const DECAY_TICK_MS = 1000; // 1 tick = 1s

// Constants
const MEADOW_COST = 200; // Flunds
const MEADOW_PRACTICE_COST = OnePracticeRun;
const MAX_PLACEMENTS = 12; // max plants to place (to avoid overwhelming)

// New plant types: define once, reuse
type PlantType = 'wildflower' | 'grass' | 'shrub' | 'groundcover' | 'legume' | 'pollinatorPlant' | 'erosionControl' | 'invasive' | 'decoy';

interface PlantConfig {
    id: PlantType;
    name: string;
    cost: number; // in meadow tokens (not city resources)
    healthStart: number;
    decayRate: number; // health/tick
    healthRegen: number; // if sunlight/water is sufficient
    icon: string; // e.g., "minigame/bloom-wildflower"
    synergy: Record<PlantType, number>; // synergy bonus per neighbor (0 to 0.25)
    effectType?: EffectType; // e.g., EffectType.LandValue, EffectType.GreenhouseGases (for city-wide effect after win)
    effectMagnitude?: number;
}

const PLANT_TYPES: PlantConfig[] = [
    {
        id: 'wildflower',
        name: 'Wildflower',
        cost: 3,
        healthStart: 100,
        decayRate: 1.5,
        healthRegen: 0.2,
        icon: "minigame/bloom-wildflower",
        synergy: { wildflower: 0.05, pollinatorPlant: 0.2, shrub: -0.05 },
        effectType: EffectType.LandValue,
        effectMagnitude: 0.02,
    },
    {
        id: 'grass',
        name: 'Fescue Grass',
        cost: 2,
        healthStart: 120,
        decayRate: 1.0,
        healthRegen: 0.4,
        icon: "minigame/bloom-grass",
        synergy: { grass: 0.1, erosionControl: 0.15, wildflower: 0.02 },
    },
    // ... 6 more types (e.g., shrub, legume, invasive thistle, decoy flower)
];

interface PlotCell {
    plantType?: PlantType;
    health: number;
    placedTick: number; // for decay calc
}

type PlotGrid = PlotCell[]; // length 36 (66)

export class BloomMinigame implements IHasDrawable, IOnResizeEvent {
    private city: City;
    private uiManager: UIManager;
    private shown = false;
    private lastDrawable?: Drawable;
    private scroller = new StandardScroller(false, true);

    // Game state
    private grid: PlotGrid = Array(GRID_SIZE * GRID_SIZE).fill(null).map(() => ({ health: 0 }));
    private gameStarted = false;
    private userInputLocked = false;
    private endReason = "";
    private winnings?: Resource[];
    private decayTimer = GAME_DURATION;
    private lastTick = 0;
    private timerTimeout?: NodeJS.Timeout;
    private preloaded = false;

    // UI state
    private selectedPlant?: PlantType;
    private howToPlayShown = false;
    private isPractice = false;
    private lastDrawable?: Drawable;

    constructor(city: City, uiManager: UIManager) {
        this.city = city;
        this.uiManager = uiManager;
    }

    // === Core Lifecycle ===

    public show(): void {
        if (this.shown) return;
        this.shown = true;
        this.preloadImages();
    }

    public hide(): void {
        this.shown = false;
        this.gameStarted = false;
        this.endReason = "";
        this.winnings = undefined;
        clearTimeout(this.timerTimeout);
    }

    public getCosts(): { type: string; amount: number }[] {
        return this.isPractice ? MEADOW_PRACTICE_COST :
            [{ type: "flunds", amount: MEADOW_COST }];
    }

    public startGame(): void {
        if (this.city.checkAndSpendResources(this.getCosts())) {
            this.gameStarted = true;
            this.userInputLocked = false;
            this.decayTimer = GAME_DURATION;
            this.lastTick = Date.now();
            this.grid = Array(GRID_SIZE * GRID_SIZE)
                .fill(null)
                .map(() => ({ health: 0 }));
            this.selectedPlant = undefined;
            this.endReason = "";
            this.winnings = undefined;

            this.startTimer();
            this.uiManager.frameRequested = true;
            this.city.updateLastUserActionTime();
            this.city.fullSave();
        }
    }

    public async preloadImages(): Promise<void> {
        if (this.preloaded) return;
        const urls: { [key: string]: string } = {
            "minigame/bloom-empty": "assets/minigame/bloom-empty.png",
            "minigame/bloom-grid": "assets/minigame/bloom-grid.png",
            "minigame/bloom-overlay": "assets/minigame/bloom-overlay.png",
        };
        for (const plant of PLANT_TYPES) {
            urls[plant.icon] = `assets/minigame/${plant.icon}.png`;
        }
        await this.uiManager.renderer.loadMoreSprites(this.city, urls);
        this.preloaded = true;
    }

    public getLastDrawable(): Drawable | null {
        return this.lastDrawable;
    }

    public onResize(): void {
        this.scroller.onResize();
    }

    public asDrawable(): Drawable {
        if (!this.shown) return (this.lastDrawable = new Drawable({ width: "0px" }));

        const main = new Drawable({
            width: "100%",
            height: "100%",
            fallbackColor: '#2d3e2a', // earthy green bg
        });

        if (!this.gameStarted) {
            this.drawStartOverlay(main);
            if (!this.howToPlayShown) this.drawCloseButton(main);
        } else {
            this.drawGameArea(main);
            if (this.userInputLocked) this.drawLockedOverlay(main);
        }

        this.lastDrawable = main;
        return main;
    }

    // === Timer Logic ===

    private startTimer(): void {
        this.timerTimeout = setTimeout(() => {
            if (!this.gameStarted) return;

            const now = Date.now();
            const elapsed = (now - this.lastTick) / 1000;
            this.lastTick = now;

            // Apply decay to all active plants
            for (let i = 0; i < this.grid.length; i++) {
                const cell = this.grid[i];
                if (cell.plantType && cell.health > 0) {
                    cell.health -= cell.healthStart * (PLANT_TYPES.find(p => p.id === cell.plantType)?.decayRate ?? 0) * elapsed;
                    if (cell.health <= 0) {
                        cell.plantType = undefined;
                        cell.health = 0;
                    }
                }
            }

            this.decayTimer -= elapsed;

            if (this.decayTimer <= 0) {
                this.endReason = "Time to observe!";
                this.endGame();
            } else {
                this.uiManager.frameRequested = true;
                this.startTimer();
            }
        }, DECAY_TICK_MS);
    }

    // === Game Logic ===

    private handleGridClick(x: number, y: number): void {
        if (this.userInputLocked) return;
        const index = y * GRID_SIZE + x;
        const cell = this.grid[index];

        if (!cell.plantType && this.selectedPlant) {
            // Place plant
            const config = PLANT_TYPES.find(p => p.id === this.selectedPlant)!;
            if (this.countPlacedPlants() >= MAX_PLACEMENTS) {
                this.notify({
                    title: "Too Full!",
                    body: "You cant plant more than " + MAX_PLACEMENTS + " plants.",
                    icon: "ui/alert"
                });
                return;
            }
            cell.plantType = this.selectedPlant;
            cell.health = config.healthStart;
            cell.placedTick = Date.now();
        } else if (cell.plantType) {
            // Remove plant (trash)
            cell.plantType = undefined;
            cell.health = 0;
        }
        this.uiManager.frameRequested = true;
    }

    private countPlacedPlants(): number {
        return this.grid.filter(c => !!c.plantType).length;
    }

    private calculateWinnings(): Resource[] {
        const activePlants = this.grid.filter(c => !!c.plantType);
        if (activePlants.length === 0) return [];

        // Calculate synergy & stability
        let synergyScore = 0;
        for (let i = 0; i < this.grid.length; i++) {
            const cell = this.grid[i];
            if (!cell.plantType) continue;
            const config = PLANT_TYPES.find(p => p.id === cell.plantType)!;
            // Check 4 neighbors (up/down/left/right)
            const neighbors = [
                i % GRID_SIZE > 0 ? this.grid[i - 1] : null,
                i % GRID_SIZE < GRID_SIZE - 1 ? this.grid[i + 1] : null,
                i >= GRID_SIZE ? this.grid[i - GRID_SIZE] : null,
                i < this.grid.length - GRID_SIZE ? this.grid[i + GRID_SIZE] : null,
            ];
            for (const n of neighbors) {
                if (n?.plantType) {
                    const bonus = config.synergy[n.plantType] ?? 0;
                    synergyScore += bonus;
                }
            }
        }

        // Stability penalty
        const coverage = activePlants.length / (GRID_SIZE * GRID_SIZE);
        let stability = 1;
        if (coverage > 0.8) stability -= 0.2; // overcrowding
        if (coverage < 0.2) stability -= 0.1; // too sparse

        const baseScore = activePlants.length * 10;
        const adjustedScore = Math.round((baseScore + synergyScore * 100) * stability);

        // Convert to rewards (simple formula)
        const flunds = Math.max(50, Math.round(adjustedScore * 1.5));
        const seeds = Math.max(1, Math.round(adjustedScore / 20)); // e.g., "Wildflower Seeds"

        const rewards: Resource[] = [
            new Flunds(flunds),
            new Paper(seeds), // reusable seeds!
        ];

        // Optional: add effect to city (only if not practice)
        if (!this.isPractice) {
            const landValueBonus = Math.round(adjustedScore / 100);
            this.city.addEffect(new Effect(
                EffectType.LandValue,
                landValueBonus * 0.01,
                undefined,
                undefined,
                Math.round(LONG_TICKS_PER_DAY * 2) // 2 days duration
            ));
        }

        return rewards;
    }

    public endGame(): void {
        this.gameStarted = false;
        clearTimeout(this.timerTimeout);
        this.winnings = this.calculateWinnings();

        // Show end screen (handled in drawStartOverlay via winningsArea)
        this.uiManager.frameRequested = true;

        if (!this.isPractice) {
            progressMinigameOptionResearch(this.city, 0.03); // ~3% chance to unlock research
        }
    }

    // === UI Drawing (Core Helpers) ===

    private drawStartOverlay(parent: Drawable): void {
        const overlay = parent.addChild(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            width: "min(100%, 560px)",
            height: "100%",
            fallbackColor: '#1b261a',
            onDrag: (x, y) => this.scroller.handleDrag(y, overlay.screenArea),
            onDragEnd: () => this.scroller.resetDrag(),
        }));

        if (this.howToPlayShown) {
            this.drawHowToPlay(overlay, parent);
            return;
        }

        let nextY = 10 - this.scroller.getScroll();
        const baseY = nextY;

        // Title
        overlay.addChild(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            y: nextY,
            width: "100%",
            height: "48px",
            text: "Bloom & Breeze",
            fontHeight: 28,
        }));
        nextY += 60;

        // Start Button
        const startBtn = overlay.addChild(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            y: nextY,
            width: "200px",
            height: "48px",
            fallbackColor: '#3e5438',
            onClick: () => this.startGame(),
            children: [new Drawable({
                anchors: ["centerX"],
                y: 5,
                width: "calc(100% - 10px)",
                height: "100%",
                text: "Start Meadow",
                centerOnOwnX: true,
            })],
        }));
        addResourceCosts(startBtn, this.getCosts(), 76, 58, false, false, false, 48, 10, 32, undefined, undefined, !this.city.hasResources(this.getCosts(), false), this.city);
        nextY += 160;

        // Practice toggle
        overlay.addChild(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            y: nextY,
            width: "500px",
            height: "48px",
            onClick: () => this.isPractice = !this.isPractice,
            children: [
                new Drawable({
                    x: 5,
                    width: "48px",
                    height: "48px",
                    image: new TextureInfo(64, 64, this.isPractice ? "ui/checked" : "ui/unchecked"),
                }),
                new Drawable({
                    anchors: ["right"],
                    rightAlign: true,
                    x: 5,
                    y: 7,
                    width: "calc(100% - 60px)",
                    height: "100%",
                    text: "Practice Run (no decay)",
                }),
            ],
        }));
        nextY += 60;

        // How to Play Button
        overlay.addChild(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            y: nextY,
            width: "200px",
            height: "48px",
            fallbackColor: '#3e5438',
            onClick: () => this.toggleRules(),
            children: [new Drawable({
                anchors: ["centerX"],
                y: 5,
                width: "calc(100% - 10px)",
                height: "100%",
                text: "How to Play",
                centerOnOwnX: true,
            })],
        }));
        nextY += 60;

        // Winnings Area
        if (this.winnings?.length) {
            const winningsArea = overlay.addChild(new Drawable({
                anchors: ["centerX"],
                centerOnOwnX: true,
                y: nextY,
                width: "min(100%, 500px)",
                height: "400px",
                fallbackColor: '#2a3d28',
                id: "winningsArea",
            }));

            winningsArea.addChild(new Drawable({
                anchors: ["centerX"],
                centerOnOwnX: true,
                biggerOnMobile: true,
                y: 10,
                width: "100%",
                height: "32px",
                text: `Biodiversity Score: ${this.winnings[0].amount / 1.5}`,
            }));

            winningsArea.addChild(new Drawable({
                anchors: ["centerX"],
                centerOnOwnX: true,
                biggerOnMobile: true,
                y: 58,
                width: "100%",
                height: "32px",
                text: "Rewards",
            }));

            const rewardContainer = winningsArea.addChild(new Drawable({
                x: 0,
                y: 100,
                width: "100%",
                fallbackColor: '#00000000',
            }));
            addResourceCosts(rewardContainer, this.winnings, 0, 0, false, false, false, 64, 10, 32, 4);
            nextY += 520;
        }

        this.scroller.setChildrenSize(nextY - baseY);
    }

    private drawGameArea(parent: Drawable): void {
        // Top bar: timer + resource hints
        let nextY = 10;
        const timerBar = parent.addChild(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            y: nextY,
            width: "min(100%, 560px)",
            height: "40px",
            fallbackColor: '#1b261a',
        }));

        // Timer bar (progress style)
        const timerWidthRatio = this.decayTimer / GAME_DURATION;
        timerBar.addChild(new Drawable({
            anchors: ["left"],
            width: (timerWidthRatio * 100) + "%",
            height: "100%",
            fallbackColor: '#2e8b57',
            image: new TextureInfo(500, 40, "ui/progressfg"),
        }));
        timerBar.addChild(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            width: "100%",
            height: "100%",
            text: `${Math.ceil(this.decayTimer)}s`,
            centerOnOwnX: true,
            fontHeight: 22,
        }));

        nextY += 50;

        // Grid
        const gridArea = parent.addChild(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            y: nextY,
            width: (GRID_SIZE * 72) + "px", // 72px per tile + margin
            height: (GRID_SIZE * 72) + "px",
            fallbackColor: '#2d3e2a',
        }));

        const tileSize = 72;
        const margin = 4;
        for (let y = 0; y < GRID_SIZE; y++) {
            for (let x = 0; x < GRID_SIZE; x++) {
                const index = y * GRID_SIZE + x;
                const cell = this.grid[index];
                const tileX = x * tileSize;
                const tileY = y * tileSize;

                const tile = gridArea.addChild(new Drawable({
                    x: tileX + margin,
                    y: tileY + margin,
                    width: tileSize - 2 * margin + "px",
                    height: tileSize - 2 * margin + "px",
                    image: new TextureInfo(tileSize, tileSize, cell.plantType ? PLANT_TYPES.find(p => p.id === cell.plantType)?.icon : "minigame/bloom-empty"),
                    onClick: () => this.handleGridClick(x, y),
                }));

                // Health bar (small red/green overlay)
                if (cell.plantType) {
                    const config = PLANT_TYPES.find(p => p.id === cell.plantType)!;
                    const healthPct = cell.health / config.healthStart;
                    tile.addChild(new Drawable({
                        anchors: ["bottom"],
                        y: 2,
                        width: "100%",
                        height: "8px",
                        fallbackColor: healthPct < 0.3 ? '#cc3333' : (healthPct < 0.6 ? '#cc9933' : '#33cc33'),
                        clipWidth: healthPct,
                    }));
                }
            }
        }

        nextY += GRID_SIZE * tileSize;

        // Plant Selector (bottom)
        nextY += 10;
        const selector = parent.addChild(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            y: nextY,
            width: (5 * tileSize) + "px",
            height: (tileSize + 10) + "px",
            fallbackColor: '#1b261a',
        }));

        let selX = 0;
        for (const plant of PLANT_TYPES.slice(0, 5)) {
            selector.addChild(new Drawable({
                x: selX,
                y: 5,
                width: tileSize + "px",
                height: tileSize + "px",
                image: new TextureInfo(tileSize, tileSize, plant.icon),
                onClick: () => {
                    this.selectedPlant = this.selectedPlant === plant.id ? undefined : plant.id;
                },
                children: [
                    new Drawable({
                        anchors: ["bottom", "centerX"],
                        y: 2,
                        width: "100%",
                        height: "24px",
                        text: `${plant.cost} bud`,
                        rightAlign: true,
                        fontHeight: 14,
                    }),
                    new Drawable({
                        anchors: ["top", "centerX"],
                        y: 2,
                        width: "100%",
                        height: "16px",
                        text: plant.name,
                        centerOnOwnX: true,
                        fontHeight: 12,
                        grayscale: this.selectedPlant !== plant.id,
                        // reddize = selected
                        reddize: this.selectedPlant === plant.id,
                    }),
                ],
            }));
            selX += tileSize + 6;
        }

        // Give Up Button (top-right corner of game area)
        parent.addChild(new Drawable({
            anchors: ["top", "right"],
            x: -10,
            y: 10,
            width: "64px",
            height: "32px",
            fallbackColor: '#cc3333',
            onClick: () => this.endGame(),
            children: [new Drawable({
                anchors: ["centerX", "centerY"],
                text: "Give Up",
                centerOnOwnX: true,
                fontHeight: 18,
            })],
        }));
    }

    private drawHowToPlay(overlay: Drawable, root: Drawable): void {
        root.onClick = () => this.toggleRules();
        let nextY = 10 - this.scroller.getScroll();

        overlay.addChild(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            y: nextY,
            width: "100%",
            height: "48px",
            text: "How to Play",
            fontHeight: 32,
        }));
        nextY += 60;

        const rules = [
            "Place native plants on the grid to build a balanced ecosystem.",
            "Plants decay over time  plant fast or use resources to preserve them.",
            " compatible plants boost each others health and yield.",
            "Avoid overcrowding (>80% coverage) or too-sparse planting (<20%).",
            "End early to give up. Youll earn Flunds + seeds based on biodiversity.",
        ];

        for (const rule of rules) {
            overlay.addChild(new Drawable({
                anchors: ["bottom"],
                y: -40,
                width: "calc(100% - 40px)",
                height: "50px",
                wordWrap: true,
                keepParentWidth: true,
                text: rule,
            }));
            nextY += 70;
        }

        this.scroller.setChildrenSize(nextY + 400); // safe estimate
    }

    private drawLockedOverlay(parent: Drawable): void {
        parent.addChild(new Drawable({
            anchors: ["centerX", "centerY"],
            centerOnOwnX: true,
            width: "100%",
            height: "100%",
            fallbackColor: '#00000088',
            onClick: () => { }, // prevent clicks
        }));
    }

    private drawCloseButton(parent: Drawable): void {
        parent.addChild(new Drawable({
            anchors: ["top", "right"],
            x: -10,
            y: 10,
            width: "48px",
            height: "48px",
            image: new TextureInfo(64, 64, "ui/close"),
            onClick: () => this.hide(),
        }));
    }

    private toggleRules(): void {
        this.howToPlayShown = !this.howToPlayShown;
        if (this.howToPlayShown) this.scroller.resetScroll();
    }

    private notify(notice: Notification): void {
        this.uiManager.pushNotification(notice);
    }
}
