﻿// BotanistBounty.ts
import { City } from "../game/City.js";
import { UIManager } from "../ui/UIManager.js";
import { Drawable } from "../ui/Drawable.js";
import { TextureInfo } from "../ui/TextureInfo.js";
import { IHasDrawable } from "../ui/IHasDrawable.js";
import { IOnResizeEvent } from "../ui/IOnResizeEvent.js";
import { StandardScroller } from "../ui/StandardScroller.js";
import { Resource } from "../game/Resource.js";
import { Flunds, getResourceType } from "../game/ResourceTypes.js";
import { addResourceCosts } from "../ui/UIUtil.js";
import { OnePracticeRun } from "./MinigameUtil.js";

// Game constants
const GRID_SIZE = 6;
const TILE_SIZE = 64;
const GAME_DURATION = 180; // seconds
const BASE_REWARD = 50;
const MAX_BIODIVERSITY = 108; // 6x6 grid * max score per plant (3)

interface PlantType {
    id: string;
    name: string;
    icon: string;
    biodiversityScore: number;
    pestRisk: number;
    compatibleNeighbors: string[];
}

interface GameState {
    grid: (PlantType | null)[][];
    selectedPlant: PlantType | null;
    score: number;
    timer: number;
    gameOver: boolean;
    pestsOutbreak: boolean;
    countdownSteps: string[];
    countdownStep: number;
}

export class BotanistBounty implements IHasDrawable, IOnResizeEvent {
    private state: GameState;
    private uiManager: UIManager;
    private scroller: StandardScroller;
    private countdownTimeout: NodeJS.Timeout | null = null;
    private gameTimeout: NodeJS.Timeout | null = null;
    private lastDrawable: Drawable | null = null;
    public shown = false;
    public gameStarted = false;
    public howToPlayShown = false;
    public isPractice = false;
    public winnings: Resource[] = [];

    // Plant definitions
    private plantTypes: PlantType[] = [
        {
            id: "tomato",
            name: "Tomato",
            icon: "minigame/tomato",
            biodiversityScore: 2,
            pestRisk: 0.3,
            compatibleNeighbors: ["basil", "marigold"]
        },
        {
            id: "basil",
            name: "Basil",
            icon: "minigame/basil",
            biodiversityScore: 3,
            pestRisk: 0.1,
            compatibleNeighbors: ["tomato", "cucumber"]
        },
        {
            id: "marigold",
            name: "Marigold",
            icon: "minigame/marigold",
            biodiversityScore: 1,
            pestRisk: 0.4,
            compatibleNeighbors: ["tomato", "onion"]
        },
        {
            id: "cucumber",
            name: "Cucumber",
            icon: "minigame/cucumber",
            biodiversityScore: 2,
            pestRisk: 0.2,
            compatibleNeighbors: ["basil", "onion"]
        },
        {
            id: "onion",
            name: "Onion",
            icon: "minigame/onion",
            biodiversityScore: 2,
            pestRisk: 0.15,
            compatibleNeighbors: ["marigold", "cucumber"]
        }
    ];

    constructor(private city: City, uiManager: UIManager) {
        this.uiManager = uiManager;
        this.scroller = new StandardScroller(false, true);
        this.resetGame();
    }

    private resetGame(): void {
        this.state = {
            grid: Array(GRID_SIZE).fill(null).map(() => Array(GRID_SIZE).fill(null)),
            selectedPlant: null,
            score: 0,
            timer: GAME_DURATION,
            gameOver: false,
            pestsOutbreak: false,
            countdownSteps: [],
            countdownStep: 0
        };
    }

    public startGame(): void {
        if (!this.city.checkAndSpendResources(this.getCosts())) return;

        this.resetGame();
        this.gameStarted = true;
        this.city.updateLastUserActionTime();
        this.city.fullSave();
        this.startCountdown();
    }

    private startCountdown(): void {
        this.state.countdownSteps = [
            "Get ready to plan your garden...",
            "Select plants from the toolbar",
            "Maximize biodiversity",
            "Watch for pest outbreaks!"
        ];
        this.state.countdownStep = 0;

        const nextStep = () => {
            if (this.state.countdownStep >= this.state.countdownSteps.length - 1) {
                this.startGameplay();
                return;
            }

            this.state.countdownStep++;
            this.uiManager.frameRequested = true;
            this.countdownTimeout = setTimeout(nextStep, 1200);
        };

        this.uiManager.frameRequested = true;
        this.countdownTimeout = setTimeout(nextStep, 1200);
    }

    private startGameplay(): void {
        this.state.countdownSteps = [];
        this.startGameTimer();
        this.uiManager.frameRequested = true;
    }

    private startGameTimer(): void {
        this.gameTimeout = setInterval(() => {
            this.state.timer--;
            this.updateGameState();
            this.uiManager.frameRequested = true;

            if (this.state.timer <= 0) {
                this.endGame();
            }
        }, 1000);
    }

    private updateGameState(): void {
        // Calculate biodiversity score
        let biodiversity = 0;
        let pestRisk = 0;

        for (let y = 0; y < GRID_SIZE; y++) {
            for (let x = 0; x < GRID_SIZE; x++) {
                const plant = this.state.grid[y][x];
                if (!plant) continue;

                biodiversity += plant.biodiversityScore;
                pestRisk += plant.pestRisk;

                // Check neighbor compatibility
                const neighbors = this.getNeighbors(x, y);
                for (const neighbor of neighbors) {
                    if (neighbor && !plant.compatibleNeighbors.includes(neighbor.id)) {
                        biodiversity -= 1; // Penalty for incompatible neighbors
                    }
                }
            }
        }

        this.state.score = Math.max(0, biodiversity);
        this.state.pestsOutbreak = pestRisk > 0.8;
    }

    private getNeighbors(x: number, y: number): PlantType[] {
        const neighbors: PlantType[] = [];
        const directions = [[-1, 0], [1, 0], [0, -1], [0, 1]];

        for (const [dx, dy] of directions) {
            const nx = x + dx;
            const ny = y + dy;
            if (nx >= 0 && nx < GRID_SIZE && ny >= 0 && ny < GRID_SIZE) {
                const plant = this.state.grid[ny][nx];
                if (plant) neighbors.push(plant);
            }
        }

        return neighbors;
    }

    public handlePlantSelection(plant: PlantType): void {
        this.state.selectedPlant = plant;
        this.uiManager.frameRequested = true;
    }

    public handleTileClick(x: number, y: number): void {
        if (!this.state.selectedPlant || this.state.gameOver) return;

        // Check if tile is empty
        if (this.state.grid[y][x]) return;

        // Place plant
        this.state.grid[y][x] = this.state.selectedPlant;
        this.updateGameState();
        this.uiManager.frameRequested = true;
    }

    public endGame(): void {
        if (this.gameTimeout) clearInterval(this.gameTimeout);
        if (this.countdownTimeout) clearTimeout(this.countdownTimeout);

        this.gameStarted = false;

        // Calculate winnings based on score
        const scoreMultiplier = Math.max(0.1, this.state.score / MAX_BIODIVERSITY);
        const baseReward = this.isPractice ? 0 : BASE_REWARD * scoreMultiplier;

        this.winnings = [
            new Flunds(Math.floor(baseReward)),
            new Resource(getResourceType("Seeds"), Math.floor(this.state.score / 10))
        ];

        // Apply special rewards for high scores
        if (this.state.score > 80) {
            this.winnings.push(new Resource(getResourceType("Fertilizer"), 5));
        }

        if (!this.isPractice) {
            this.city.transferResourcesFrom(
                this.winnings.map(r => ({ type: r.type.id, amount: r.amount })),
                "earn"
            );

            // Check achievements
            if (this.state.score >= MAX_BIODIVERSITY) {
                this.city.checkAndAwardAchievement("botanist_perfection");
            }
        }

        this.uiManager.frameRequested = true;
    }

    public asDrawable(): Drawable {
        if (!this.shown) return this.lastDrawable = new Drawable({ width: "0px" });

        const root = new Drawable({
            width: "100%",
            height: "100%",
            fallbackColor: "#2C3E50"
        });

        if (!this.gameStarted) {
            this.drawStartScreen(root);
        } else if (this.state.countdownSteps.length > 0) {
            this.drawCountdownScreen(root);
        } else {
            this.drawGameplay(root);
        }

        this.lastDrawable = root;
        return root;
    }

    private drawStartScreen(parent: Drawable): void {
        // Title
        parent.addChild(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            y: 30,
            width: "80%",
            height: "50px",
            text: "Botanist's Bounty",
            font: "bold 32px Arial",
            textBaseline: "top"
        }));

        // How to play button
        parent.addChild(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            y: 100,
            width: "200px",
            height: "40px",
            fallbackColor: "#3498DB",
            onClick: () => this.toggleRules(),
            children: [
                new Drawable({
                    anchors: ["centerX"],
                    centerOnOwnX: true,
                    y: 8,
                    width: "100%",
                    text: "How to Play",
                    font: "16px Arial"
                })
            ]
        }));

        // Start button
        const startButton = parent.addChild(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            y: 160,
            width: "200px",
            height: "50px",
            fallbackColor: "#27AE60",
            onClick: () => this.startGame(),
            children: [
                new Drawable({
                    anchors: ["centerX"],
                    centerOnOwnX: true,
                    y: 12,
                    width: "100%",
                    text: "Start Garden",
                    font: "18px Arial"
                })
            ]
        }));

        addResourceCosts(startButton, this.getCosts(), 10, 10, false, false, false);

        // Practice mode toggle
        parent.addChild(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            y: 230,
            width: "200px",
            height: "30px",
            fallbackColor: "transparent",
            onClick: () => {
                this.isPractice = !this.isPractice;
                this.uiManager.frameRequested = true;
            },
            children: [
                new Drawable({
                    x: 0,
                    y: 0,
                    width: "30px",
                    height: "30px",
                    image: new TextureInfo(30, 30, this.isPractice ? "ui/checked" : "ui/unchecked")
                }),
                new Drawable({
                    x: 40,
                    y: 5,
                    width: "calc(100% - 40px)",
                    height: "20px",
                    text: "Practice Run (no rewards)",
                    font: "14px Arial"
                })
            ]
        }));

        // Last winnings display
        if (this.winnings.length > 0) {
            const winningsArea = parent.addChild(new Drawable({
                anchors: ["centerX"],
                centerOnOwnX: true,
                y: 280,
                width: "min(100%, 500px)",
                height: "120px",
                fallbackColor: "#34495E",
                children: [
                    new Drawable({
                        anchors: ["centerX"],
                        centerOnOwnX: true,
                        y: 10,
                        width: "250px",
                        height: "25px",
                        text: `Last Score: ${this.state.score}/${MAX_BIODIVERSITY}`,
                        font: "bold 16px Arial"
                    }),
                    new Drawable({
                        x: 20,
                        y: 45,
                        width: "calc(100% - 40px)",
                        height: "60px",
                        fallbackColor: "transparent",
                        children: [
                            new Drawable({
                                x: 0,
                                y: 0,
                                width: "60px",
                                height: "60px",
                                image: new TextureInfo(60, 60, "resource/flunds")
                            }),
                            new Drawable({
                                x: 70,
                                y: 15,
                                width: "calc(100% - 70px)",
                                height: "30px",
                                text: `${this.winnings[0].amount} Flunds`,
                                font: "16px Arial"
                            })
                        ]
                    })
                ]
            }));
        }
    }

    private drawCountdownScreen(parent: Drawable): void {
        parent.addChild(new Drawable({
            anchors: ["centerX", "centerY"],
            centerOnOwnX: true,
            centerOnOwnY: true,
            width: "300px",
            height: "150px",
            fallbackColor: "rgba(0,0,0,0.7)",
            children: [
                new Drawable({
                    anchors: ["centerX"],
                    centerOnOwnX: true,
                    y: 40,
                    width: "250px",
                    height: "50px",
                    text: this.state.countdownSteps[this.state.countdownStep],
                    font: "bold 28px Arial",
                    textBaseline: "middle"
                })
            ]
        }));
    }

    private drawGameplay(parent: Drawable): void {
        // Draw grid
        const gridY = 80;
        for (let y = 0; y < GRID_SIZE; y++) {
            for (let x = 0; x < GRID_SIZE; x++) {
                const tile = parent.addChild(new Drawable({
                    x: 50 + x * (TILE_SIZE + 10),
                    y: gridY + y * (TILE_SIZE + 10),
                    width: TILE_SIZE + "px",
                    height: TILE_SIZE + "px",
                    image: new TextureInfo(TILE_SIZE, TILE_SIZE, "ui/tile"),
                    fallbackColor: "#34495E",
                    onClick: () => this.handleTileClick(x, y)
                }));

                const plant = this.state.grid[y][x];
                if (plant) {
                    tile.addChild(new Drawable({
                        anchors: ["centerX", "centerY"],
                        width: "80%",
                        height: "80%",
                        image: new TextureInfo(TILE_SIZE, TILE_SIZE, plant.icon)
                    }));
                }
            }
        }

        // Draw plant selector
        const selectorY = gridY + GRID_SIZE * (TILE_SIZE + 10) + 30;
        this.plantTypes.forEach((plant, index) => {
            parent.addChild(new Drawable({
                x: 50 + index * (TILE_SIZE + 15),
                y: selectorY,
                width: TILE_SIZE + "px",
                height: TILE_SIZE + "px",
                image: new TextureInfo(TILE_SIZE, TILE_SIZE, plant.icon),
                fallbackColor: this.state.selectedPlant?.id === plant.id ? "#F39C12" : "#7F8C8D",
                onClick: () => this.handlePlantSelection(plant)
            }));
        });

        // Draw timer
        parent.addChild(new Drawable({
            anchors: ["right"],
            x: -30,
            y: 30,
            width: "200px",
            height: "30px",
            fallbackColor: "#E74C3C",
            text: `Time: ${this.state.timer}s`
        }));

        // Draw score
        parent.addChild(new Drawable({
            anchors: ["right"],
            x: -30,
            y: 70,
            width: "200px",
            height: "30px",
            fallbackColor: "#2ECC71",
            text: `Biodiversity: ${this.state.score}/${MAX_BIODIVERSITY}`
        }));

        // Pest outbreak warning
        if (this.state.pestsOutbreak) {
            parent.addChild(new Drawable({
                anchors: ["centerX"],
                centerOnOwnX: true,
                y: 30,
                width: "80%",
                height: "40px",
                fallbackColor: "#E74C3C",
                text: "⚠️ Pest outbreak! Plant incompatible species!",
                font: "bold 18px Arial"
            }));
        }

        // Give up button
        parent.addChild(new Drawable({
            anchors: ["right"],
            x: -30,
            y: 110,
            width: "100px",
            height: "30px",
            fallbackColor: "#95A5A6",
            onClick: () => this.endGame(),
            text: "Give Up"
        }));
    }

    private toggleRules(): void {
        this.howToPlayShown = !this.howToPlayShown;
        if (this.howToPlayShown) {
            this.scroller.resetScroll();
        }
        this.uiManager.frameRequested = true;
    }

    private drawHowToPlay(overlay: Drawable): void {
        overlay.addChild(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            y: 20,
            width: "80%",
            height: "40px",
            text: "How to Play: Botanist's Bounty",
            font: "bold 24px Arial"
        }));

        const rules = [
            "1. Select plants from the bottom toolbar",
            "2. Click on empty tiles to plant them",
            "3. Compatible neighbors increase biodiversity",
            "4. Incompatible neighbors decrease biodiversity",
            "5. Too many risky plants cause pest outbreaks",
            "6. Earn rewards based on final biodiversity score"
        ];

        let nextY = 80;
        rules.forEach(rule => {
            overlay.addChild(new Drawable({
                anchors: ["centerX"],
                centerOnOwnX: true,
                y: nextY,
                width: "80%",
                height: "30px",
                text: rule,
                font: "16px Arial"
            }));
            nextY += 40;
        });

        overlay.addChild(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            y: nextY + 20,
            width: "200px",
            height: "40px",
            fallbackColor: "#3498DB",
            onClick: () => {
                this.howToPlayShown = false;
                this.uiManager.frameRequested = true;
            },
            text: "Got it!",
            font: "16px Arial"
        }));
    }

    private getCosts(): { type: string, amount: number }[] {
        return this.isPractice ? OnePracticeRun :
            [{ type: "Seeds", amount: 10 }, { type: "Flunds", amount: 200 }];
    }

    public onResize(): void {
        this.scroller.onResize();
    }

    public async preloadImages(): Promise<void> {
        // Load plant textures
        const urls: Record<string, string> = {};
        this.plantTypes.forEach(plant => {
            urls[plant.icon] = `assets/${plant.icon}.png`;
        });

        // Load UI elements
        urls["ui/tile"] = "assets/ui/tile.png";
        urls["ui/checked"] = "assets/ui/checked.png";
        urls["ui/unchecked"] = "assets/ui/unchecked.png";

        await this.uiManager.renderer.loadMoreSprites(this.city, urls);
    }

    public getLastDrawable(): Drawable | null {
        return this.lastDrawable;
    }
}
