﻿// ──────────────────────────────────────────────────────────────
// 1️⃣  Imports – everything the spec already exports
// ──────────────────────────────────────────────────────────────
import { UIManager } from "../ui/UIManager.js";
import { City } from "../game/City.js";
import { IHasDrawable } from "../ui/IHasDrawable.js";
import { IOnResizeEvent } from "../ui/IOnResizeEvent.js";
import { Drawable } from "../ui/Drawable.js";
import { StandardScroller } from "../ui/StandardScroller.js";
import { TextureInfo } from "../ui/TextureInfo.js";
import { Notification } from "../game/Notification.js";
import {
    OnePracticeRun,
    progressMinigameOptionResearch,
    rangeMapLinear,
} from "./MinigameUtil.js";

import {
    addResourceCosts,
    humanizeFloor,
    longTicksToDaysAndHours,
} from "../ui/UIUtil.js";

import { Resource } from "../game/Resource.js";
import { Flunds } from "../game/ResourceTypes.js";

// ──────────────────────────────────────────────────────────────
// 2️⃣  Game‑specific constants (feel free to tweak)
// ──────────────────────────────────────────────────────────────
const TILE = "48px";               // one tile size used everywhere
const TIMER_TICK = 1000;             // ms per timer tick (1 s)
const MAX_ORDERS = 8;                // max orders visible at once
const ORDER_HEIGHT = 48;             // px height of an order row
const ORDER_WIDTH = 200;             // px width of an order row
const ORDER_GAP = 10;                // spacing between rows
const MAX_TIMER = 120;               // seconds – game ends when it reaches 0

/** All mutable game state for Logi‑Rush */
class LogiRushState {
    // ── Core runtime ────────────────────────────────────────
    gameStarted = false;               // true after the player hits “Start”
    timer = MAX_TIMER;                 // seconds left – counts down each tick
    timerTimeout: number | null = null; // NodeJS.Timeout handle (for cleanup)

    // ── Orders ───────────────────────────────────────────────
    /** One order = { id, type, reward } */
    orders: Order[] = [];
    nextOrderId = 0;

    // ── Player‑controlled trucks ─────────────────────────────
    /** Trucks the player can drag onto an order */
    trucks: Truck[] = [];

    // ── UI / Interaction flags ───────────────────────────────
    userInputLocked = false;           // true while an animation or intro runs
    isPractice = false;                // true when the player chose “Practice Run”
    howToPlayShown = false;            // toggles the rule overlay
    winnings: Resource[] = [];         // last completed playthrough rewards
    lastDrawable: Drawable | null = null; // cached drawable from asDrawable()
    shown = false;                     // used by the base class to avoid re‑draw
    scroller: StandardScroller;        // scroll container for long rule text

    // ── Misc ─────────────────────────────────────────────────
    /** For timers that need a reference (animations, etc.) */
    animateTimeout: number | null = null;

    /** Helper – last reason the game ended (used only once) */
    endReason = "";
}
export type Order = {
    id: number;
    /** e.g. "Electronics", "Food", "Construction Materials" */
    type: string;
    /** How many units the player must deliver */
    amount: number;
    /** Reward in Flunds (or any other resource) */
    reward: number;
};

export type Truck = {
    /** The drag‑source Drawable that represents a truck */
    node: Drawable;
    /** Which order the truck is currently assigned to (null = free) */
    assignedTo: Order | null;
};

/** ---------------------------------------------------------------
 *  Logi‑Rush – a logistics routing minigame
 *
 *  Implements the two required interfaces:
 *   • IHasDrawable – provides `asDrawable()` for UI rendering
 *   • IOnResizeEvent – forwards resize events to the scroller
 * --------------------------------------------------------------- */
export class LogiRush implements IHasDrawable, IOnResizeEvent {
    // -----------------------------------------------------------------
    // 3️⃣  Dependencies & members
    // -----------------------------------------------------------------
    private readonly city: City;
    private readonly uiManager: UIManager;
    private readonly state: LogiRushState = new LogiRushState();

    /** The cached drawable returned by the last `asDrawable()` call */
    private lastDrawable: Drawable | null = null;

    /** Used by the framework to know that a redraw is needed */
    uiManager: UIManager; // re‑assigned in ctor – see below

    // -----------------------------------------------------------------
    // 4️⃣  Constructor
    // -----------------------------------------------------------------
    constructor(city: City, uiManager: UIManager) {
        this.city = city;
        this.uiManager = uiManager;

        // Make the scroller available everywhere (it will hold the scrollable UI)
        this.state.scroller = new StandardScroller(false, true); // vertical only

        // Bind the resize handler once (so we can reuse it in `onResize`)
        this.onResize = this.onResize.bind(this);
    }

    // -----------------------------------------------------------------
    // 5️⃣  Interface: IOnResizeEvent
    // -----------------------------------------------------------------
    onResize(): void {
        // The scroller must be resized whenever the window changes size.
        this.state.scroller.onResize();
    }

    // -----------------------------------------------------------------
    // 6️⃣  Interface: IHasDrawable
    // -----------------------------------------------------------------
    /** Main entry point – builds the whole UI tree and returns the root Drawable */
    asDrawable(): Drawable {
        // ---------------------------------------------------------------
        // 6.1  Cache handling – first call creates a fresh root Drawable
        // ---------------------------------------------------------------
        if (!this.state.shown) {
            // The spec says: always start with a zero‑size placeholder
            this.lastDrawable = new Drawable({ width: "0px" });
            return this.lastDrawable;
        }

        // ---------------------------------------------------------------
        // 6.2  Root container – full‑screen, magenta background so we see bugs
        // ---------------------------------------------------------------
        const root = new Drawable({
            width: "100%",
            height: "100%",
            fallbackColor: "#111111",
        });

        // ---------------------------------------------------------------
        // 6.3  Attach the scrollable container (covers the whole screen)
        // ---------------------------------------------------------------
        const overlay = root.addChild(
            new Drawable({
                anchors: ["centerX"],
                centerOnOwnX: true,
                width: "min(100%, 600px)",
                height: "100%",
                fallbackColor: "#222222",
                // The scroller will be our “scrollable canvas”.  All UI elements are added to it.
                children: [this.state.scroller],
            })
        );

        // ---------------------------------------------------------------
        // 6.4  Decide whether we show the start‑screen or the actual game UI
        // ---------------------------------------------------------------
        if (!this.state.gameStarted) {
            this.drawStartOverlay(overlay);
        } else {
            this.drawGameArea(overlay);
        }

        // ---------------------------------------------------------------
        // 6.5  Store the final drawable for the “getLastDrawable()” cache
        // ---------------------------------------------------------------
        this.lastDrawable = root;
        return root;
    }

    // -----------------------------------------------------------------
    // 7️⃣  Helper: Start‑Screen UI (shown when !gameStarted)
    // -----------------------------------------------------------------
    private drawStartOverlay(parent: Drawable): void {
        // ---------------------------------------------------------------
        // 7.1  Create a full‑screen background that also works as a drag
        //      surface for the scroller (so we can scroll the rule text later)
        // ---------------------------------------------------------------
        const bg = parent.addChild(
            new Drawable({
                anchors: ["centerX"],
                centerOnOwnX: true,
                width: "min(100%, 600px)",
                height: "100%",
                fallbackColor: "#111111",
                // Drag handling – forwards to the scroller so the text can be moved
                onDrag: (x: number, y: number) =>
                    this.state.scroller.handleDrag(y, bg.screenArea),
                onDragEnd: () => this.state.scroller.resetDrag(),
            })
        );

        // ---------------------------------------------------------------
        // 7.2  Title
        // ---------------------------------------------------------------
        bg.addChild(
            new Drawable({
                anchors: ["centerX"],
                centerOnOwnX: true,
                y: 20,
                width: "100%",
                height: "48px",
                text: "Logi‑Rush",
                fontSize: "28px",
                textAlign: "center",
                fallbackColor: "#ffffff",
            })
        );

        // ---------------------------------------------------------------
        // 7.3  “How to Play” button (toggles the rule overlay)
        // ---------------------------------------------------------------
        const howToBtn = bg.addChild(
            new Drawable({
                anchors: ["centerX"],
                centerOnOwnX: true,
                y: 100,
                width: "260px",
                height: "48px",
                fallbackColor: "#444444",
                onClick: () => this.toggleHowToPlay(),
            })
        );
        howToBtn.addChild(
            new Drawable({
                anchors: ["centerX"],
                y: 5,
                width: "calc(100% - 10px)",
                height: "100%",
                text: "How to Play",
                centerOnOwnX: true,
            })
        );

        // ---------------------------------------------------------------
        // 7.4  “Practice Run” toggle (no rewards)
        // ---------------------------------------------------------------
        const practiceChk = bg.addChild(
            new Drawable({
                anchors: ["centerX"],
                centerOnOwnX: true,
                y: 170,
                width: "300px",
                height: "48px",
                fallbackColor: "#444444",
                onClick: () => (this.state.isPractice = !this.state.isPractice),
            })
        );
        practiceChk.addChild(
            new Drawable({
                x: 5,
                width: "48px",
                height: "48px",
                image: new TextureInfo(64, 64, this.state.isPractice ? "ui/checked" : "ui/unchecked"),
            })
        );
        practiceChk.addChild(
            new Drawable({
                anchors: ["right"],
                rightAlign: true,
                x: 5,
                y: 7,
                width: "calc(100% - 60px)",
                height: "100%",
                text: "Practice Run (no rewards)",
            })
        );

        // ---------------------------------------------------------------
        // 7.5  “Start Game” button – disabled if the player cannot afford it
        // ---------------------------------------------------------------
        const cost = this.getCosts(); // returns [{type:string,amount:number}]
        const startBtn = bg.addChild(
            new Drawable({
                anchors: ["centerX"],
                centerOnOwnX: true,
                y: 260,
                width: "260px",
                height: "48px",
                fallbackColor: "#444444",
                onClick: () => this.startGame(),
            })
        );
        startBtn.addChild(
            new Drawable({
                anchors: ["centerX"],
                y: 5,
                width: "calc(100% - 10px)",
                height: "100%",
                text: "Start Game",
                centerOnOwnX: true,
            })
        );

        // Visual cue for unaffordable start‑cost
        const unaffordable = !this.city.checkAndSpendResources(cost, false);
        addResourceCosts(startBtn, cost, 86, 58, false, false, false, 48, 10, 32, undefined, undefined, unaffordable, this.city);

        // ---------------------------------------------------------------
        // 7.6  Optional “Close / Exit” button (only shown when rules are open)
        // ---------------------------------------------------------------
        if (this.state.howToPlayShown) {
            const closeBtn = bg.addChild(
                new Drawable({
                    anchors: ["right"],
                    rightAlign: true,
                    x: 10,
                    y: 10,
                    width: "48px",
                    height: "48px",
                    fallbackColor: "#ff4444",
                    onClick: () => this.toggleHowToPlay(),
                })
            );
            closeBtn.addChild(
                new Drawable({
                    x: 12,
                    y: 12,
                    width: "32px",
                    height: "32px",
                    image: new TextureInfo(64, 64, "ui/cross"),
                })
            );
        }

        // ---------------------------------------------------------------
        // 7.7  Finish up – tell the scroller how tall its content is
        // ---------------------------------------------------------------
        // Rough estimate – the UI above is ~500 px tall; we give a buffer.
        this.state.scroller.setChildrenSize(600);
    }

    // -----------------------------------------------------------------
    // 8️⃣  Helper: Toggle the rule overlay (how‑to‑play)
    // -----------------------------------------------------------------
    private toggleHowToPlay(): void {
        this.state.howToPlayShown = !this.state.howToPlayShown;
        if (this.state.howToPlayShown) {
            // Reset scroll position whenever we open the rules again
            this.state.scroller.resetScroll();
        }
        // Force a redraw so the close‑button appears/disappears correctly
        this.uiManager.frameRequested = true;
    }

    // -----------------------------------------------------------------
    // 9️⃣  Helper: Build the actual gameplay UI (shown after start)
    // -----------------------------------------------------------------
    private drawGameArea(parent: Drawable): void {
        // ---------------------------------------------------------------
        // 9.1  The scrollable canvas – we need a deterministic height
        // ---------------------------------------------------------------
        const gameArea = parent.addChild(
            new Drawable({
                anchors: ["centerX"],
                centerOnOwnX: true,
                width: "min(100%, 600px)",
                height: "100%",
                fallbackColor: "#222222",
                children: [this.state.scroller],
            })
        );

        // ---------------------------------------------------------------
        // 9.2  Helper to compute the next Y coordinate for a new child.
        //      All draw helpers return the *next* Y value so we can chain them.
        // ---------------------------------------------------------------
        let nextY = 10 - this.state.scroller.getScroll();

        // ---------------------------------------------------------------
        // 9.3  Draw the order list (scrollable)
        // ---------------------------------------------------------------
        nextY = this.drawOrdersSection(gameArea, nextY);

        // ---------------------------------------------------------------
        // 9.4  Draw the timer bar at the top
        // ---------------------------------------------------------------
        nextY = this.drawTimerBar(gameArea, nextY);

        // ---------------------------------------------------------------
        // 9.5  Draw any “extra actions” (e.g., a “Reset” button)
        // ---------------------------------------------------------------
        nextY = this.drawExtraButtons(gameArea, nextY);

        // ---------------------------------------------------------------
        // 9.6  Tell the scroller its new content height
        // ---------------------------------------------------------------
        this.state.scroller.setChildrenSize(nextY - (10 - this.state.scroller.getScroll()));
    }

    // -----------------------------------------------------------------
    // 10️⃣  Sub‑helper: Orders list (the heart of the game)
    // -----------------------------------------------------------------
    private drawOrdersSection(parent: Drawable, startY: number): number {
        // The orders sit inside a scrollable container that we own.
        const orderContainer = new Drawable({
            x: 0,
            y: startY,
            width: "100%",
            height: "calc(100% - 150px)", // leave room for timer & buttons below
            // No explicit background – we rely on the scroller’s own background.
        });

        // ---------------------------------------------------------------
        // 10.1  Populate orders (re‑generate every frame if the timer is ticking)
        // ---------------------------------------------------------------
        if (!this.state.gameStarted || this.state.timer % 15 === 0) {
            this.regenerateOrders();
        }

        // ---------------------------------------------------------------
        // 10.2  Render each order as a small row
        // ---------------------------------------------------------------
        const maxRows = Math.floor((this.state.scroller.getScroll() + 600) / (ORDER_HEIGHT + ORDER_GAP));
        const visibleOrders = this.state.orders.slice(0, maxRows);

        visibleOrders.forEach((ord, idx) => {
            const yPos = idx * (ORDER_HEIGHT + ORDER_GAP);
            const row = orderContainer.addChild(
                new Drawable({
                    x: 0,
                    y: yPos,
                    width: ORDER_WIDTH,
                    height: ORDER_HEIGHT,
                    fallbackColor: "#333333",
                    // Slight hover effect – change background colour
                    onHover: () => (row.fallbackColor = "#555555"),
                    onLeave: () => (row.fallbackColor = "#333333"),
                })
            );

            // Order title
            row.addChild(
                new Drawable({
                    anchors: ["left"],
                    x: 10,
                    y: ORDER_HEIGHT / 2 - 12,
                    width: "calc(100% - 20px)",
                    height: "12px",
                    text: `${ord.type} (x${ord.amount})`,
                    fontSize: "14px",
                    rightAlign: false,
                })
            );

            // Reward indicator (small coin icon)
            row.addChild(
                new Drawable({
                    x: ORDER_WIDTH - 30,
                    y: ORDER_HEIGHT / 2 - 12,
                    width: "16px",
                    height: "16px",
                    image: new TextureInfo(64, 64, "ui/coin"),
                })
            );

            // -----------------------------------------------------------
            // 10.3  Drag source – the player can drag a truck onto this row
            // -----------------------------------------------------------
            row.onClick = () => this.handleOrderClick(ord, row);
        });

        // ---------------------------------------------------------------
        // 10.4  Return the Y coordinate after the last order row
        // ---------------------------------------------------------------
        return startY + (visibleOrders.length * (ORDER_HEIGHT + ORDER_GAP));
    }

    // -----------------------------------------------------------------
    // 11️⃣  Order generation – simple random generator for demo purposes
    // -----------------------------------------------------------------
    private regenerateOrders(): void {
        const types = ["Electronics", "Food", "Construction Materials", "Clothing"];
        const newOrders: Order[] = [];

        const count = Math.min(MAX_ORDERS, Math.floor(Math.random() * 3) + 1);
        for (let i = 0; i < count; i++) {
            const type = types[Math.floor(Math.random() * types.length)];
            const amount = Math.max(1, Math.floor(Math.random() * 4) + 1);
            const reward = Math.max(5, Math.floor(Math.random() * 6) + 5);
            newOrders.push({ id: this.state.nextOrderId++, type, amount, reward });
        }
        this.state.orders = newOrders;
    }

    // -----------------------------------------------------------------
    // 12️⃣  Click handling for an order row
    // -----------------------------------------------------------------
    private handleOrderClick(order: Order, row: Drawable): void {
        // If the player clicks an empty slot we just ignore it – we only
        // allow dragging a truck onto it.
        if (!this.state.trucks.some((t) => t.assignedTo === null)) {
            // All trucks are busy – give a quick visual feedback
            row.fallbackColor = "#ff2222";
            setTimeout(() => (row.fallbackColor = "#333333"), 300);
            return;
        }

        // Find a free truck (the first one we can assign)
        const freeTruck = this.state.trucks.find((t) => t.assignedTo === null);
        if (!freeTruck) return;

        // Assign the truck to this order
        freeTruck.assignedTo = order;

        // Visually move the truck node onto the row (so the player sees it)
        const truckNode = freeTruck.node;
        truckNode.anchors = []; // remove any previous anchors
        truckNode.x = 0;
        truckNode.y = 0;
        truckNode.width = ORDER_WIDTH;
        truckNode.height = ORDER_HEIGHT;
        truckNode.fallbackColor = "#00ff88";

        // Store the assignment on the truck object so we can later free it
        // (the assignment itself is enough; we only need the reference for UI)
    }

    // -----------------------------------------------------------------
    // 13️⃣  Timer bar – drawn at the top of the screen
    // -----------------------------------------------------------------
    private drawTimerBar(parent: Drawable, startY: number): number {
        const timerBar = parent.addChild(
            new Drawable({
                anchors: ["centerX"],
                centerOnOwnX: true,
                y: startY,
                width: "min(100%, 300px)",
                height: "30px",
                fallbackColor: "#444444",
                children: [
                    new Drawable({
                        clipWidth: 0, // will be updated each frame
                        width: "100%",
                        height: "100%",
                        fallbackColor: "#00ff11",
                        image: new TextureInfo(200, 20, "ui/progressbg"),
                        children: [
                            new Drawable({
                                clipWidth: (this.state.timer / MAX_TIMER) * 0.94, // 0‑1 ratio
                                width: "100%",
                                height: "100%",
                                fallbackColor: "#00ff11",
                                image: new TextureInfo(200, 20, "ui/progressfg"),
                                reddize: this.state.timer < 5,
                            }),
                        ],
                    }),
                ],
            })
        );

        // Show remaining seconds as text next to the bar
        const txt = timerBar.addChild(
            new Drawable({
                anchors: ["right"],
                rightAlign: true,
                x: -10,
                y: 4,
                width: "80px",
                height: "100%",
                text: `${Math.ceil(this.state.timer)}s`,
                fontSize: "14px",
                textAlign: "center",
            })
        );

        // Return the Y coordinate *after* the timer bar so further elements
        // can be stacked underneath it.
        return startY + 40; // height of timer bar + small margin
    }

    // -----------------------------------------------------------------
    // 14️⃣  Extra UI – Reset button, close‑rules button, etc.
    // -----------------------------------------------------------------
    private drawExtraButtons(parent: Drawable, startY: number): number {
        const btnY = startY + 10;
        const resetBtn = parent.addChild(
            new Drawable({
                anchors: ["centerX"],
                centerOnOwnX: true,
                y: btnY,
                width: "180px",
                height: "48px",
                fallbackColor: "#555555",
                onClick: () => this.resetGame(),
            })
        );
        resetBtn.addChild(
            new Drawable({
                anchors: ["centerX"],
                y: 5,
                width: "calc(100% - 10px)",
                height: "100%",
                text: "Reset Game",
                centerOnOwnX: true,
            })
        );

        // Close‑rules button – only visible when the overlay is open
        const closeBtn = parent.addChild(
            new Drawable({
                anchors: ["right"],
                rightAlign: true,
                x: 10,
                y: 10,
                width: "48px",
                height: "48px",
                fallbackColor: "#ff4444",
                onClick: () => this.toggleHowToPlay(),
            })
        );
        closeBtn.addChild(
            new Drawable({
                x: 12,
                y: 12,
                width: "32px",
                height: "32px",
                image: new TextureInfo(64, 64, "ui/cross"),
            })
        );

        return btnY + 60; // a bit of margin for the next possible UI
    }

    // -----------------------------------------------------------------
    // 15️⃣  Game flow – start, tick, end
    // -----------------------------------------------------------------
    /** Called when the player presses “Start Game”. */
    private startGame(): void {
        const cost = this.getCosts();
        if (!this.city.checkAndSpendResources(cost, false)) {
            // Not enough resources – already shown by addResourceCosts()
            return;
        }

        // ---------------------------------------------------------------
        // 15.1  Initialise state
        // ---------------------------------------------------------------
        this.state.gameStarted = true;
        this.state.timer = MAX_TIMER;
        this.state.orders = [];
        this.state.trucks = [];

        // ---------------------------------------------------------------
        // 15.2  Create a few *default* trucks the player can drag.
        //        In a full game you would probably have a pool of trucks
        //        stored somewhere else, but for this demo we just make
        //        three simple placeholder trucks.
        // ---------------------------------------------------------------
        for (let i = 0; i < 3; i++) {
            const truck = {
                node: new Drawable({
                    width: TILE,
                    height: TILE,
                    fallbackColor: "#5555ff",
                    image: new TextureInfo(64, 64, "ui/truck"),
                }),
                assignedTo: null,
            };
            this.state.trucks.push(truck);
        }

        // ---------------------------------------------------------------
        // 15.3  Start the countdown tick
        // ---------------------------------------------------------------
        this.tick();

        // ---------------------------------------------------------------
        // 15.4  Force a redraw now that the UI has changed
        // ---------------------------------------------------------------
        this.uiManager.frameRequested = true;
    }

    /** Tick the timer once per second. */
    private tick(): void {
        // Decrement the timer
        this.state.timer--;
        // Request a redraw so the progress bar updates
        this.uiManager.frameRequested = true;

        if (this.state.timer <= 0) {
            // -----------------------------------------------------------
            // 15.4.1  Time's up – end the game
            // -----------------------------------------------------------
            this.state.endReason = "Time's up!";
            this.endGame();
            return;
        }

        // Schedule the next tick
        this.state.timerTimeout = setTimeout(() => this.tick(), TIMER_TICK);
    }

    /** Called automatically when the timer hits zero or the player quits. */
    private endGame(): void {
        // ---------------------------------------------------------------
        // 15.5  Compute rewards based on how many orders were fulfilled
        // ---------------------------------------------------------------
        const fulfilled = this.state.trucks.filter((t) => t.assignedTo !== null);
        const rewardMultiplier = this.state.isPractice ? 0 : 1;
        const totalReward = fulfilled.reduce((sum, t) => sum + (t.assignedTo!.reward * rewardMultiplier), 0);

        // ---------------------------------------------------------------
        // 15.6  Convert the raw reward into actual Resource objects
        //        (the spec uses `new Flunds(amount)` for currency)
        // ---------------------------------------------------------------
        const winnings: Resource[] = [];
        if (totalReward > 0) {
            winnings.push(new Flunds(totalReward));
        }
        this.state.winnings = winnings;

        // ---------------------------------------------------------------
        // 15.7  Fire the end‑game hook – the base class may push an event,
        //        give the player a badge, etc.
        // ---------------------------------------------------------------
        this.endReason = "Finished";
        this.handleGameEnd();
    }

    /** Central place where the minigame can push notifications, rewards, etc. */
    private handleGameEnd(): void {
        // Example: award a “Logi‑Rush Master” badge if the player earned > 200 Flunds
        if (this.state.winnings.some((r) => r.amount > 200)) {
            this.city.notify(
                new Notification("Logi‑Rush Master", "You earned a huge haul!", "ui/medal")
            );
        }

        // Reset UI flags for a possible next run
        this.state.gameStarted = false;
        this.state.timer = MAX_TIMER;
        this.state.timerTimeout && clearTimeout(this.state.timerTimeout);
        this.state.timerTimeout = null;

        // Force a redraw of the start screen
        this.uiManager.frameRequested = true;
    }

    /** Called when the player wants to start over (reset button). */
    private resetGame(): void {
        // Simple confirmation – in a real game you’d open a modal
        this.state.endReason = "Reset by player";
        this.handleGameEnd();
    }

    // -----------------------------------------------------------------
    // 16️⃣  Utility – cost definition (used by start‑screen & practice toggle)
    // -----------------------------------------------------------------
    /** Returns an array of `{type:string, amount:number}` describing the price. */
    private getCosts(): { type: string; amount: number }[] {
        // Practice runs cost *nothing* except the optional “OnePracticeRun” flag
        if (this.state.isPractice) {
            return []; // free
        }

        // Normal run – charge a flat fee of 500 Flunds (you can expose more fields)
        return [{ type: "Flunds", amount: 500 }];
    }

    // -----------------------------------------------------------------
    // 17️⃣  Practice‑run flag handling (used by the toggle button)
    // -----------------------------------------------------------------
    private togglePractice(): void {
        // This method is wired to the “Practice Run” checkbox in drawStartOverlay()
        // – we keep it tiny because the UI already flips the boolean.
        // If you need to adjust costs, do it here.
    }

    // -----------------------------------------------------------------
    // 18️⃣  Pre‑load assets – called once by the framework before the first draw
    // -----------------------------------------------------------------
    public async preloadImages(): Promise {
        if (this.state.preloaded) return; // already loaded

        const assetMap: Record = {
            "ui/truck": "assets/ui/truck.png",
            "ui/coin": "assets/ui/coin.png",
            "ui/cross": "assets/ui/cross.png",
            "ui/checked": "assets/ui/checked.png",
            "ui/unchecked": "assets/ui/unchecked.png",
            "ui/progressbg": "assets/ui/progressbg.png",
            "ui/progressfg": "assets/ui/progressfg.png",
            // Minigame‑specific sprites
            "minigame/emptyTrash": "assets/minigame/emptyTrash.png",
            "minigame/trash_full": "assets/minigame/trash_full.png",
        };

        await this.uiManager.renderer.loadMoreSprites(this.city, assetMap);
        this.state.preloaded = true;
    }

    // -----------------------------------------------------------------
    // 19️⃣  Public getter required by the base framework
    // -----------------------------------------------------------------
    public getLastDrawable(): Drawable | null {
        return this.lastDrawable;
    }
}
