﻿import { IHasDrawable, IOnResizeEvent } from "../ui/IHasDrawable";
import { Drawable } from "../ui/Drawable";
import { StandardScroller } from "../ui/StandardScroller";
import { UIManager } from "../ui/UIManager";
import { Resource } from "../game/Resource";
import { Flunds, getResourceType } from "../game/ResourceTypes";
import { TextureInfo } from "../ui/TextureInfo";
import { addResourceCosts, humanizeFloor, longTicksToDaysAndHours } from "../ui/UIUtil";
import { OnePracticeRun, progressMinigameOptionResearch, rangeMapLinear } from "../MinigameUtil.js";
import { GameState } from "../game/GameState";
import { inPlaceShuffle } from "../game/MiscFunctions";
import { EffectType, WasteEffect } from "../game/GridType"; // Custom effect for waste decay

// Constants (easy to modify)
const GAME_DURATION = 90; // seconds
const COMPOST_DURATION = 45; // seconds per decomposition stage
const MAX_BINS = 3; // Mobile-friendly limit
const BIN_SPACING = 180; //px between bins (adapts to screen size)
const WASTE_YIELD_TABLE: Record<string, number> = {
    "apple core": 0.8,
    "vegetable peel": 1.2,
    "coffee grounds": 2.0,
    "plastic bottle": 0, // Non-compostable
    "paper cup": 0.5,
    "cardboard": 0.3
};

// Soil/compost compatibility (for visual feedback)
const SOIL_TYPES = {
    "fertile": "#4CAF50",   // Green
    "poor": "#9E9E9E",      // Gray
    "acidic": "#FF9800",    // Orange
    "alkaline": "#E91E63"    // Pink
};

interface WasteItem {
    type: string;
    value: number; // Base decomposition time multiplier
    color: string;
    icon: string;
}

interface CompostCycleState {
    currentYear: number; // Game year context
    activeBins: Map<number, { contents: WasteItem[], fullProgress: number }>; // Bin index -> state
    wasteItems: (WasteItem[])[]; // Global inventory pool
    decompositionProgress: Map<number, number>; // Item index -> progress (0-100%)
    startTime: number | null;
    currentCycleStage: "collection" | "decomposition" | "fertilization" | "end";
    playerScore: number;
    gameEnded: boolean;
    endReason: string | null;
    // UI State
    showHowTo: boolean;
    showTips: boolean;
    decompositionTimer: NodeJS.Timeout | null;
}

export class CompostCycleMinigame implements IHasDrawable, IOnResizeEvent {
    private state: CompostCycleState;
    private uiManager: UIManager;
    private scroller: StandardScroller;
    private binOverlays: Drawable[] = [];
    private tipsContainer: Drawable;
    private tipsText: string[];
    private lastDrawable: Drawable = new Drawable({ width: "0px" });

    constructor(
        city: City,
        uiManager: UIManager,
        private cityGrid: Grid { farmTiles: Tile[] } // Pre-filtered farm tiles
    ) {
        this.uiManager = uiManager;
        this.scroller = this.uiManager.scroller;
        this.state = initializeState(this.cityGrid);
        this.binOverlays = [];
        this.tipsContainer = null;
        this.tipsText = ["Tap items to add to compost", "Long-press to skip stage", "Rotate bins to change soil type"];
    }

    // Core state initialization (copies farm tiles + empty waste bins)
    private initializeState(farmGrid: Grid) {
        const { farmTiles } = farmGrid;
        const tileCount = farmTiles.length;

        return {
            currentYear: 1,
            activeBins: new Map(
                Array(MAX_BINS).fill().map(() => ({
                    contents: [],
                    fullProgress: 0,
                    allocatedItems: 3 // Max items per bin
                }))
            ),
            wasteItems: this.generateWasteItems(),
            decompositionProgress: new Map(
                Array(tileCount).fill().map(() => 0)
            ),
            startTime: null,
            currentCycleStage: "collection",
            playerScore: 0,
            gameEnded: false,
            showHowTo: false,
            showTips: false,
            decompositionTimer: null,
            // Preload assets
            binAssets: {
                empty: "assets/minigame/bin-empty.png",
                full: "assets/minigame/bin-full.png",
                decomposing: "assets/minigame/compost-stage.png"
            }
        };
    }

    private generateWasteItems(): WasteItem[] {
        // 3-5 random waste types per playthrough (vary by difficulty)
        const itemsPerBin = Math.floor(rangeMapLinear(3, 5, "mid") * (0.7 + 0.6 * (this.state.difficulty || 0.5)));
        const selectedTypes = new Set<string>();
        while (selectedTypes.size < Math.floor(itemsPerBin)) {
            selectedTypes.add(WASTE_YIELD_TABLE[Object.keys(WASTE_YIELD_TABLE)[Math.floor(Math.random() * Object.keys(WASTE_YIELD_TABLE).length)]]);
        }
        return Array.from(selectedTypes).map(type => ({
            ...WASTE_YIELD_TABLE[type],
            color: SOIL_TYPES[Object.values(SOIL_TYPES)[Math.floor(Math.random() * Object.values(SOIL_TYPES).length)]],
            icon: `icon/${Object.keys(WASTE_YIELD_TABLE)[Object.keys(WASTE_YIELD_TABLE).find(k => WASTE_YIELD_TABLE[k] === type)]}.png`
        })).slice(0, itemsPerBin);
    }

    public startCycle(): void {
        if (this.state.gameEnded) return;

        this.state.startTime = Date.now();
        this.state.currentCycleStage = "collection";
        this.state.playerScore = 0;
        this.resetBins();

        // Add tutorial tips
        this.showTips(true);
        setTimeout(() => this.showTips(false), 5000);

        this.uiManager.frameRequested = true;
    }

    private resetBins(): void {
        this.binOverlays = [];
        this.state.activeBins.forEach((_, binIndex) => {
            this.state.activeBins.get(binIndex)!.allocatedItems = 3;
        });
    }

    private handleBinClick(binIndex: number, x: number, y: number): void {
        if (this.state.gameEnded || this.state.currentCycleStage !== "collection") return;

        const binBounds = this.getBinBounds(binIndex);
        if (!insideRectangle(x, y, binBounds.x, binBounds.y, binBounds.width, binBounds.height)) {
            return;
        }

        // Find nearest waste item (raycasting simplified)
        const nearestItem = this.state.wasteItems.reduce((closest, item) => {
            const dx = x - (item.x || 0);
            const dy = y - (item.y || 0);
            return (dx * dx + dy * dy) < closest.distance ? { ...closest, distance: dx * dx + dy * dy } : closest;
        }, { distance: Infinity });

        if (!nearestItem) return;

        if (this.canPlaceItem(nearestItem, binIndex)) {
            this.placeWasteItem(nearestItem, binIndex);
            this.updateScore(nearestItem.value);
            this.uiManager.frameRequested = true;
        }
    }

    private canPlaceItem(item: WasteItem, binIndex: number): boolean {
        const bin = this.state.activeBins.get(binIndex);
        if (!bin) return false;

        // Capacity check (3 items max)
        if (bin.allocatedItems >= 3) return false;

        // Decomposition time check (prevent backlogs)
        const binFull = bin.contents.some(c => c.progress >= 95 %);
        if (binFull && item.value > 60) return false; // Skip large items

        return true;
    }

    private placeWasteItem(item: WasteItem, binIndex: number): void {
        const bin = this.state.activeBins.get(binIndex)!;
        bin.contents.push(item);
        bin.allocatedItems++;
        bin.progress = 10; // Start decomposition immediately

        // Update visual feedback
        this.updateBinVisual(binIndex, true);
        this.state.decompositionProgress.set(item.id!, 0); // Track by global item ID
    }

    private updateScore(itemValue: number): void {
        const yieldBonus = WASTE_YIELD_TABLE[itemValue.type] * (0.8 + 0.2 * this.state.difficulty);
        this.playerScore += yieldBonus;
        this.state.playerScore += yieldBonus;
    }

    private startDecomposition(): void {
        if (this.state.gameEnded) return;

        this.state.currentCycleStage = "decomposition";
        this.state.startTime = Date.now();

        // Start per-bin timers
        this.state.activeBins.forEach(bin => {
            const progressInterval = setInterval(() => {
                if (Date.now() - this.state.startTime > COMPOST_DURATION * bin.progress) {
                    bin.progress += 15;
                    clearInterval(progressInterval);
                    return;
                }
            }, 100);

            this.state.decompositionProgress.set(bin.contents[0]?.id || 0,
                Math.min(100, this.state.decompositionProgress.get(bin.contents[0]?.id || 0) + 0.5));

            // Animate visual progress
            this.animateBin(binIndex, bin.progress);
        }, COMPOST_DURATION);
    }

    private startFertilization(): void {
        if (this.state.gameEnded) return;

        this.state.currentCycleStage = "fertilization";
        let remainingTime = 30; // Fertilization duration

        setInterval(() => {
            if (remainingTime <= 0) {
                this.completeCycle();
                return;
            }

            // Calculate fertilizer yield based on bin contents
            let totalYield = 0;
            this.state.activeBins.forEach(bin => {
                if (bin.progress >= 90) { // Only count fully decomposed bins
                    bin.contents.forEach(item => {
                        totalYield += WASTE_YIELD_TABLE[item.type] * (0.7 + 0.3 * this.state.difficulty);
                    });
                }
            });

            this.state.playerScore += totalYield;
            remainingTime -= 5;

            // Visual feedback
            this.updateFertilizerBar();
            this.uiManager.frameRequested = true;

            // Auto-advance stage after 3 bins finish
            if (this.countCompleteBins() >= this.state.activeBins.length && remainingTime <= 0) {
                this.state.currentCycleStage = "end";
                this.startTimer();
            }
        }, 200);
    }

    private countCompleteBins(): number {
        return this.state.activeBins.filter(bin => bin.progress >= 90).length;
    }

    private completeCycle(): void {
        this.state.gameEnded = true;
        this.state.startTime = Date.now();
        this.state.currentCycleStage = "end";

        // Apply city-wide effect: Temporary fertility boost
        const bonusEffect = new Effect({
            type: EffectType.LandValue,
            multiplier: 1.2,
            building: undefined,
            dynamicCalculation: "getTileFertility", // Custom city function
            expirationLongTicks: LONG_TICKS_PER_DAY * 2 // 2 day duration
        });
        this.city.effects.push(bonusEffect);

        // Calculate winnings
        const winnings = [
            new Flunds(this.playerScore * 0.15), // Direct currency reward
            new Electronics(Math.floor(this.playerScore * 0.05)), // Secondary reward
            new Apples(Math.floor(this.playerScore * 0.03)) // Tertiary reward
        ];
        this.calculateWinnings(winnings); // Converts to city-compatible rewards

        this.showResults(this.playerScore, winnings);
    }

    public startTimer(): void {
        if (this.state.decompositionTimer) clearTimeout(this.state.decompositionTimer);

        this.state.decompositionTimer = setTimeout(() => {
            if (!this.state.gameEnded) {
                this.startFertilization();
            }
        }, GAME_DURATION * 1000);
    }

    // UI Rendering Methods (asDrawable decomposition)
    private asDrawable(): Drawable {
        if (!this.shown) return this.lastDrawable = new Drawable({ width: "0px" });

        const mainDrawable = new Drawable({
            width: "100%",
            height: "100%",
            fallbackColor: "#222222",
            anchorX: "left",
            anchorY: "top"
        });

        // Scroller setup
        this.scroller.onResize();
        this.updateScrollerSize();

        if (this.state.gameEnded) {
            this.drawResults(mainDrawable);
            return mainDrawable;
        }

        if (this.state.currentCycleStage === "collection") {
            this.drawCollectionScreen(mainDrawable);
        } else if (this.state.currentCycleStage === "decomposition") {
            this.drawDecompositionScreen(mainDrawable);
        } else {
            this.drawFertilizationScreen(mainDrawable);
        }

        this.lastDrawable = mainDrawable;
        return mainDrawable;
    }

    private drawCollectionScreen(parent: Drawable): void {
        const baseY = 80;
        let nextY = baseY;

        // Bin containers
        this.binOverlays = this.activeBins.keys().map((binIndex: number) => {
            const bin = this.activeBins.get(binIndex)!;
            const yOffset = (binIndex * (BIN_SPACING + 20)) + 100;

            const binDrawable = parent.addChild(new Drawable({
                x: 20,
                y: nextY,
                width: "calc(100% - 40px)",
                height: "120px",
                falloCust: true,
                falloCustMode: 0.85,
                biggerOnMobile: false,
                grayscale: false,
                scaleYOnMobile: true,
                fallbackColor: "#555555",
                image: new TextureInfo(160, 24, this.state.binAssets.empty),
                children: [
                    new Drawable({
                        width: "100%",
                        height: "100%",
                        falloCust: false,
                        falloCustMode: 1,
                        x: 10,
                        y: 5,
                        scaleX: 1,
                        scaleY: 1,
                        image: new TextureInfo(150, 30, "ui/fill-ratio"),
                        centerOnOwnX: true
                    }),
                    new Drawable({
                        width: "100%",
                        height: "100%",
                        falloCust: false,
                        falloCustMode: 1,
                        x: 10,
                        y: 40,
                        scaleX: 1,
                        scaleY: 1,
                        image: new TextureInfo(150, 30, "ui/empty-ratio"),
                        centerOnOwnX: true
                    })
                ]
            })
    });

        nextY += 150;

        // Waste items panel
        this.tipsContainer = parent.addChild(new Drawable({
            x: 20,
            y: nextY - this.state.scroller.getScroll(),
            width: "calc(100% - 40px)",
            height: "120px",
            falloCust: true,
            falloCustMode: 0.85,
            biggerOnMobile: false,
            grayscale: false,
            scaleYOnMobile: true,
            fallbackColor: "#555555",
            text: this.state.tipsText[0],
            wordWrap: true,
            keepParentWidth: true,
            centerOnOwnX: true,
            onClick: () => this.startCycle(),
            onLongPress: () => {
                this.state.currentCycleStage = "decomposition"; // Skip to decomposition
                this.uiManager.frameRequested = true;
            }
        }));

        nextY += 80;
        this.renderWasteItems(nextY, parent);

        // Bin controls
        const startButton = this.binOverlays[0].addChild(new Drawable({
            x: 0,
            y: 0,
            width: "100%",
            height: "40px",
            fallbackColor: "#666666",
            onClick: () => this.startCycle(),
            children: [
                new Drawable({ text: "New Cycle", rightAlign: true, centerOnOwnX: true })
            ]
        }));

        // ... (additional UI elements: progress bar, timer, tips)

        this.scroller.setChildrenSize(nextY - baseY);
    }

    private drawDecompositionScreen(parent: Drawable): void {
        const baseY = 80;
        let nextY = baseY;

        // Progress bar header
        this.binOverlays.forEach((_, binIndex) => {
            const binDrawable = parent.addChild(new Drawable({
                x: 20,
                y: nextY - this.state.scroller.getScroll(),
                width: "calc(100% - 40px)",
                height: "20px",
                falloCust: true,
                falloCostMode: 0.85,
                biggerOnMobile: false,
                grayscale: false,
                scaleYOnMobile: true,
                fallbackColor: "#AAAAAA",
                anchorX: "left",
                anchorY: "top",
                anchorSelf: ["left", "top"],
                selfCenterX: true,
                selfCenterY: true,
                selfWidth: 0,
                x: 0,
                y: 0
            })
                .thenDraw(ctx => {
                    const progress = this.activeBins.get(binIndex)!.progress || 0;
                    const barWidth = (progress / 100) * 196; // 20px bar + 2px padding
                    ctx.clearRect(0, 0, 196, 20);
                    ctx.fillStyle = progress > 80 ? "#4CAF50" : "#9E9E9E";
                    ctx.fillRect(0, 0, barWidth, 20);
                })
                .addTo(parent);

            nextY += 220;
        });

        // Global timer
        this.renderTimer(nextY, parent);
        nextY += 60;

        // Waste inventory preview
        this.tipsContainer = parent.addChild(new Drawable({
            x: 20,
            y: nextY - this.state.scroller.getScroll(),
            width: "calc(100% - 40px)",
            height: "120px",
            falloCust: true,
            falloCustMode: 0.85,
            biggerOnMobile: false,
            grayscale: false,
            scaleYOnMobile: true,
            fallbackColor: "#555555",
            text: "Items left: " + this.state.wasteItems.length,
            wordWrap: true,
            keepParentWidth: true,
            centerOnOwnX: true
        }));

        nextY += 80;
        this.renderWasteItems(nextY, parent, true); // Visual items only

        this.scroller.setChildrenSize(nextY - baseY);
    }

    private renderWasteItems(startY: number, parent: Drawable, showIcons: boolean = false): void {
        const itemWidth = 60;
        const padding = 15;

        this.state.wasteItems.forEach((item, i) => {
            const x = (i * (itemWidth + padding)) + 30;
            const y = startY + 20;

            const itemDrawable = showIcons
                ? new Drawable({
                    x, y,
                    width: itemWidth,
                    height: 60,
                    image: new TextureInfo(item.color || "#FFFFFF", 36, item.icon),
                    onClick: () => this.handleWasteClick(i) // Delegate to handleWasteClick(itemIndex)
                })
                : new Drawable({ // Text-only version
                    x, y,
                    width: "100%",
                    height: "100%",
                    text: item.type.toUpperCase() + " →",
                    fontSize: 14,
                    yAlign: "center",
                    centerOnOwnX: true,
                    noXStretch: true
                });

            parent.addChild(itemDrawable);
        });
    }

    private renderTimer(startY: number, parent: Drawable): void {
        const hours = Math.floor(GAME_DURATION / 3600);
        const minutes = Math.floor((GAME_DURATION % 3600) / 60);
        const seconds = GAME_DURATION % 60;

        const timerDrawable = parent.addChild(new Drawable({
            x: 20,
            y: startY,
            width: "calc(100% - 40px)",
            height: "30px",
            falloCust: true,
            falloCustMode: 0.85,
            biggerOnMobile: false,
            grayscale: false,
            scaleYOnMobile: true,
            fallbackColor: "#AAAAAA",
            id: "timer"
        })
            .thenDraw(ctx => {
                const elapsed = Date.now() - this.state.startTime;
                const secondsLeft = GAME_DURATION - elapsed;

                ctx.clearRect(0, 0, 196, 30);

                // Format time string
                let timeString = "";
                if (hours > 0) timeString += `${hours}h `;
                if (minutes > 0) timeString += `${minutes}m `;
                timeString += `${seconds}s`;

                // Animate countdown bar
                const barWidth = (secondsLeft / GAME_DURATION) * 196;
                ctx.fillStyle = secondsLeft < 10 ? "#FF0000" : "#4CAF50";
                ctx.fillRect(0, 0, barWidth, 30);

                // Display text
                ctx.fillStyle = "#FFFFFF";
                ctx.font = "8px sans-serif";
                ctx.textAlign = "left";
                if (timeString) ctx.fillText(timeString, 10, 12);
            })
            .addTo(parent);

        // Drag handling for timer container
        const container = timerDrawable.children?.[0] || timerDrawable;
        container.onDrag = (x, y) => this.scroller.handleDrag(y, container.screenArea);
        container.onDragEnd = () => this.scroller.resetDrag();
    }

    // Custom draw bin visual states
    private animateBin(binIndex: number, progress: number): void {
        const binDrawable = this.binOverlays[binIndex];
        if (!binDrawable || this.state.gameEnded) return;

        const fillRatio = 1 - (1 - progress / 100) * 0.7; // Color shift based on progress
        binDrawable
            .children[0]
            .image
            .texture = new TextureInfo(150, 30, "ui/fill-" + Math.floor(fillRatio * 10) % 10); // 0-9 fill states

        this.uiManager.frameRequested = true;
    }

    private updateFertilizerBar(): void {
        const completeBins = this.countCompleteBins();
        const barWidth = (completeBins / MAX_BINS) * 196;

        const fertilizerBar = this.binOverlays.find(b => !b.id) || this.binOverlays[0];
        fertilizerBar
            .children[0]
            .image
            .texture = new TextureInfo(196, 20, "ui/fertile-bar");
      .clipWidth = barWidth;

        if (completeBins >= Math.floor(MAX_BINS * 0.7)) {
            fertilizerBar.children[1].image.texture = new TextureInfo(30, 20, "ui/glow");
        }
    }

    // Results display with winnings visualization
    private drawResults(score: number, winnings: Resource[]): void {
        const baseY = 80;
        let nextY = baseY;

        // Title and score
        const title = nextY -= 20;
        new Drawable({
            x: 20,
            y: title - this.state.scroller.getScroll(),
            width: "calc(100% - 40px)",
            height: "40px",
            text: "Cycle Complete!",
            fontSize: 16,
            centerOnOwnX: true
        }).addTo(this.lastDrawable);

        nextY += 25;
        new Drawable({
            x: 20,
            y: nextY - this.state.scroller.getScroll(),
            width: "calc(100% - 40px)",
            height: "30px",
            text: `Score: ${Math.floor(score * 100)}`,
            fontSize: 14,
            centerOnOwnX: true
        }).addTo(this.lastDrawable);
        nextY += 30;

        // Winnings panel
        this.lastDrawable.addChild(new Drawable({
            id: "winnings",
            x: 20,
            y: nextY - this.state.scroller.getScroll(),
            width: "calc(100% - 40px)",
            height: "300px",
            falloCust: true,
            falloCustMode: 0.85,
            biggerOnMobile: false,
            grayscale: false,
            scaleYOnMobile: true,
            fallbackColor: "#555555",
            children: winnings.map((win: Resource, i: number) => {
                const amount = win.amount || 0;
                const color = getResourceColor(win.type, amount > 0);

                return new Drawable({
                    x: 10,
                    y: i * (60 + 10),
                    width: "calc(70% - 20px)",
                    height: "25px",
                    falloCust: false,
                    falloCustMode: 1,
                    image: new TextureInfo(70, 20, color),
                    text: `${win.type} × ${Math.floor(amount * 10)}`,
                    fontSize: 12,
                    textColor: "#FFFFFF",
                    wordWrap: true,
                    keepParentWidth: true,
                    centerOnOwnX: true
                });
            })
        })
            .addTo(this.lastDrawable);

        // Practice mode toggle
        nextY += 340;
        this.lastDrawable.addChild(new Drawable({
            x: 20,
            y: nextY - this.state.scroller.getScroll(),
            width: "100%",
            height: "60px",
            falloCust: true,
            falloCustMode: 0.85,
            biggerOnMobile: false,
            grayscale: false,
            scaleYOnMobile: true,
            onClick: () => {
                if (this.state.isPractice) {
                    this.state.isPractice = false;
                    this.startCycle();
                } else {
                    // Create practice mode with infinite bins/waste
                    this.createPracticeMode();
                }
            }
        }));

        this.scroller.setChildrenSize(nextY - baseY);
    }

    private createPracticeMode(): void {
        // Duplicate state with infinite resources
        const practiceState = JSON.parse(JSON.stringify(this.state));
        practiceState.activeBins = this.state.activeBins.map(bin => ({ ...bin, progress: 0 }));
        practiceState.wasteItems = [...this.state.wasteItems, ...this.generateWasteItems()];
        practiceState.startTime = Date.now();
        practiceState.currentCycleStage = "collection";
        practiceState.gameEnded = false;

        // Clone to separate state instance
        // (Implementation detail - would use state pattern)
        this.state = practiceState;
        this.startCycle();
    }

    // User interaction handlers
    private handleWasteClick(itemIndex: number): void {
        if (this.state.currentCycleStage !== "collection") return;

        const item = this.state.wasteItems[itemIndex];
        if (!this.canPlaceItem(item, 0)) return; // Bin 0 for testing

        this.placeWasteItem(item, 0);
        this.uiManager.frameRequested = true;
    }

    private handleBinDragEnd(x: number, y: number): void {
        if (this.state.currentCycleStage === "decomposition") return;

        // Convert screen position to bin coordinates
        const binWidth = 196; // 100px visible + padding
        const binIndex = Math.floor((this.scroller.scrollY + y) / binWidth);

        // Only process if user stopped dragging over a bin
        if (binIndex < this.binOverlays.length) {
            this.handleBinClick(binIndex, x, y);
        }
    }

    // Utility functions
    private getResourceColor(type: string, amount: number): string {
        const baseColor = getResourceBaseColor(type);
        return amount > 10 ? baseColor + "80" : baseColor; // Dim for small amounts
    }

    private getResourceBaseColor(type: string): string {
        const COLORS: Record<string, string> = {
            "Flunds": "#FFD700",
            "Apples": "#FF9900",
            "Electronics": "#00BFFF",
            "Copper": "#B87333",
            "Waste": "#4A4A4A"
        };
        return COLORS[type] || "#CCCCCC";
    }

    public preloadImages(): Promise<void> {
        // Minigame-specific assets
        const images = {
            "compost-bin-empty": "assets/minigame/bin-empty.png",
            "compost-bin-full": "assets/minigame/bin-full.png",
            "compost-stage-1": "assets/minigame/compost-stage1.png",
            "compost-stage-2": "assets/minigame/compost-stage2.png",
            "compost-stage-3": "assets/minigame/compost-stage3.png",
            "fill-0": "ui/fill-0",
            "fill-1": "ui/fill-1",
            "fill-2": "ui/fill-2",
            "fill-3": "ui/fill-3",
            "fill-4": "ui/fill-4",
            "fill-5": "ui/fill-5",
            "fill-6": "ui/fill-6",
            "fill-7": "ui/fill-7",
            "fill-8": "ui/fill-8",
            "fill-9": "ui/fill-9",
            "fertile-bar": "ui/fertile-bar",
            "glow": "ui/glow",
            "fill-ratio": "ui/fill-ratio",
            "empty-ratio": "ui/empty-ratio"
        };

        // Load assets
        await this.uiManager.loadMoreSprites(this.city, images);
        return Promise.resolve();
    }

    public onResize(): void {
        this.scroller.onResize();
    }

    public hide(): void {
        this.state.shown = false;
        this.lastDrawable = new Drawable({ width: "0px" });
    }

    public show(): void {
        this.startCycle();
    }

    private calculateWinnings(winnings: Resource[]): void {
        // Convert minigame winnings to city-compatible events
        this.city.events = this.city.events.filter(e => e.type !== "compostCycle");

        winnings.forEach(win => {
            switch (win.type) {
                case "Flunds":
                    this.city.addCoins(new Flunds(win.amount));
                    break;
                case "Electronics":
                    this.city.unlockBuilding(this.city.getBuildingType("Factory"));
                    break;
                case "Apples":
                    this.city.checkAndAwardAchievement(`"Epic Farmer"`);
                    break;
            }
        });
    }
}
