import { TitleTypes } from "../game/AchievementTypes.js";
import { City } from "../game/City.js";
import { CityFlags } from "../game/CityFlags.js";
import { ProductionReward, TourismReward } from "../game/EventTypes.js";
import { LONG_TICKS_PER_DAY } from "../game/FundamentalConstants.js";
import { UIManager } from "../ui/UIManager.js";
import { Resource, Flunds, getResourceType } from "../game/Resource.js";
import { TextureInfo } from "../ui/TextureInfo.js";
import { Drawable } from "../ui/Drawable.js";
import { addResourceCosts, humanizeFloor, longTicksToDaysAndHours } from "../ui/UIUtil.js";
import { IHasDrawable } from "../ui/IHasDrawable.js";
import { IOnResizeEvent } from "../ui/IOnResizeEvent.js";
import { StandardScroller } from "../ui/StandardScroller.js";
import { inPlaceShuffle } from "../game/MiscFunctions.js";
import { GameState } from "../game/GameState.js";
import { drawMinigameOptions } from "../ui/MinigameOptions.js";
import { OnePracticeRun, progressMinigameOptionResearch, rangeMapLinear } from "./MinigameUtil.js";
import { EffectType } from "../game/GridType.js";

// Configuration constants
const GAME_DURATION = 120;
const INITIAL_SPAWN_DELAY = 3000;
const SPAWN_INTERVAL = 5000;
const PACKAGE_TIME_TO_LIVE = 45;
const BASE_PACKAGE_POINTS = 100;
const TIME_BONUS_PER_SECOND = 10;
const PERFECT_DELIVERY_BONUS = 50;
const DIRECT_DELIVERY_MULTIPLIER = 1.5;
const PERFECT_DELIVERY_MULTIPLIER = 1.25;
const MAX_COMBO = 5;
const TILE_SIZE = 64;
const GRID_WIDTH = 12;
const NUM_PICKUP_ZONES = 4;
const NUM_DELIVERY_ZONES = 4;
const ZONE_SIZE = 2;
const ZONE_SPACING = 3;
const ZONE_PADDING = 8;
const MAX_PACKAGES_PER_ZONE = 3;
const NUM_VEHICLES = 2;
const VEHICLE_CAPACITY = 4;
const VEHICLE_SIZE = 48;
const VEHICLE_SPEED = 0.5;
const VEHICLE_OFFSET = 8;
const PACKAGE_SIZE = 32;
const PACKAGE_OFFSET = 16;
const PROGRESS_BAR_WIDTH = 300;
const MIN_SCORE = 500;
const MAX_SCORE = 5000;
const MIN_REWARD_AMOUNT = 1;
const MAX_REWARD_AMOUNT = 5;

type PackageStatus = "pending_spawn" | "available" | "picked_up" | "loaded" | "in_transit" | "delivered" | "expired";

interface Package {
    id: string;
    pickupZone: LocationZone;
    destinationZone: LocationZone;
    status: PackageStatus;
    points: number;
    timeToLive?: number;
    size: "small" | "medium" | "large";
    color: string;
    vehicleId?: string;
    screenX?: number;
    screenY?: number;
}

interface LocationZone {
    id: string;
    type: "pickup" | "delivery" | "vehicle";
    gridX: number;
    gridY: number;
    width: number;
    height: number;
    maxPackages?: number;
    packages: Package[];
    vehicle?: DeliveryVehicle;
}

class DeliveryVehicle {
    id: string;
    currentZone: LocationZone | null;
    targetZone: LocationZone | null;
    currentLoad: Package[] = [];
    maxCapacity: number;
    moveSpeed: number;
    isMoving: boolean = false;
    arrivalTime: number = 0;

    constructor(id: string, maxCapacity: number, moveSpeed: number) {
        this.id = id;
        this.maxCapacity = maxCapacity;
        this.moveSpeed = moveSpeed;
    }

    loadPackage(pkg: Package): boolean {
        if (this.currentLoad.length >= this.maxCapacity) return false;
        this.currentLoad.push(pkg);
        pkg.status = "loaded";
        pkg.vehicleId = this.id;
        return true;
    }

    setDestination(zone: LocationZone): void {
        this.targetZone = zone;
        if (this.currentZone && zone) {
            const distance = this.calculateDistance(this.currentZone, zone);
            const travelTime = (distance / this.moveSpeed) * 1000;
            this.arrivalTime = Date.now() + travelTime;
            this.isMoving = true;
        }
    }

    private calculateDistance(from: LocationZone, to: LocationZone): number {
        const dx = to.gridX - from.gridX;
        const dy = to.gridY - from.gridY;
        return Math.sqrt(dx * dx + dy * dy);
    }
}

class ParcelPatrolState {
    score: number = 0;
    timeBonus: number = 0;
    comboMultiplier: number = 1;
    perfectDeliveries: number = 0;
    currentLevel: number = 1;
    timer: number = GAME_DURATION;
    gameStarted: boolean = false;
    userInputLocked: boolean = false;
    endReason: string = "";
    packages: Package[] = [];
    vehicles: DeliveryVehicle[] = [];
    pickupZones: LocationZone[] = [];
    deliveryZones: LocationZone[] = [];
    selectedPackage: Package | null = null;
    dragOffsetX: number = 0;
    dragOffsetY: number = 0;
    highlightedZone: LocationZone | null = null;
    vehicleFullMessageShown: boolean = false;
    nextSpawnTime: number = 0;
    packagesPerSpawn: number = 3;
    countdownSteps: string[] = ["3", "2", "1", "GO!"];
    countdownStep: number = 0;
}

export class ParcelPatrol implements IHasDrawable, IOnResizeEvent {
    private state: ParcelPatrolState;
    private zoneDrawables: Map<string, Drawable> = new Map();
    private vehicleDrawables: Map<string, Drawable> = new Map();
    private packageDrawables: Map<string, Drawable> = new Map();
    private scroller: StandardScroller;
    private timerTimeout: NodeJS.Timeout | null = null;
    private spawnTimeout: NodeJS.Timeout | null = null;
    private lastDrawable: Drawable | null = null;
    public shown: boolean = false;
    public isPractice: boolean = false;
    private winnings: Resource[] = [];
    private lastScore: number = 0;
    private lastPerfectDeliveries: number = 0;
    private spendResearch: boolean = false;

    constructor(private city: City, private uiManager: UIManager) {
        this.state = new ParcelPatrolState();
        this.scroller = new StandardScroller(false, true);
    }

    private createZones(): void {
        for (let i = 0; i < NUM_PICKUP_ZONES; i++) {
            this.state.pickupZones.push({
                id: `pickup_${i}`,
                type: "pickup",
                gridX: i % 2 === 0 ? 0 : ZONE_SPACING,
                gridY: Math.floor(i / 2) * ZONE_SPACING,
                width: ZONE_SIZE,
                height: ZONE_SIZE,
                maxPackages: MAX_PACKAGES_PER_ZONE,
                packages: []
            });
        }
        for (let i = 0; i < NUM_DELIVERY_ZONES; i++) {
            this.state.deliveryZones.push({
                id: `delivery_${i}`,
                type: "delivery",
                gridX: GRID_WIDTH - ZONE_SIZE,
                gridY: i * ZONE_SPACING,
                width: ZONE_SIZE,
                height: ZONE_SIZE,
                packages: []
            });
        }
    }

    private createVehicles(): void {
        for (let i = 0; i < NUM_VEHICLES; i++) {
            const vehicle = new DeliveryVehicle(`vehicle_${i}`, VEHICLE_CAPACITY, VEHICLE_SPEED);
            vehicle.currentZone = this.state.pickupZones[0];
            this.state.vehicles.push(vehicle);
        }
    }

    public show(): void {
        if (this.shown) return;
        this.shown = true;
        this.initializeGame();
        this.uiManager.frameRequested = true;
    }

    private initializeGame(): void {
        this.state = new ParcelPatrolState();
        this.createZones();
        this.createVehicles();
        this.state.nextSpawnTime = Date.now() + INITIAL_SPAWN_DELAY;
    }

    private getCosts(): { type: string; amount: number }[] {
        return this.isPractice ? OnePracticeRun :
            this.spendResearch ? [{ type: "research", amount: 10 }, { type: "flunds", amount: 200 }] :
                [{ type: "flunds", amount: 1000 }];
    }

    public startGame(): void {
        const costs = this.getCosts();
        if (!this.city.checkAndSpendResources(costs)) {
            this.showInsufficientResourcesMessage();
            return;
        }
        this.setInputLocked(true);
        this.state.gameStarted = true;
        this.state.countdownSteps = ["3", "2", "1", "GO!"];
        this.state.countdownStep = 0;
        this.runCountdown();
        this.city.updateLastUserActionTime();
        this.game.fullSave();
    }

    private runCountdown(): void {
        if (this.state.countdownStep >= this.state.countdownSteps.length) {
            this.setInputLocked(false);
            this.startTimer();
            this.startPackageSpawning();
            return;
        }
        const step = this.state.countdownSteps[this.state.countdownStep];
        this.showCountdownNumber(step);
        this.state.countdownStep++;
        setTimeout(() => this.runCountdown(), 1000);
    }

    private startTimer(): void {
        this.timerTimeout = setTimeout(() => {
            if (!this.state.gameStarted || this.state.userInputLocked) return;
            this.state.timer--;
            if (this.state.timer <= 0) {
                this.endReason = "Time's up!";
                this.endGame();
            } else {
                this.startTimer();
            }
            this.uiManager.frameRequested = true;
        }, 1000);
    }

    private startPackageSpawning(): void {
        this.spawnPackages();
    }

    private spawnPackages(): void {
        if (!this.state.gameStarted || this.state.userInputLocked) return;
        const availableZones = this.state.pickupZones.filter(z => z.packages.length < (z.maxPackages || 3));
        if (availableZones.length === 0) {
            this.state.nextSpawnTime = Date.now() + 2000;
            return;
        }
        for (let i = 0; i < this.state.packagesPerSpawn; i++) {
            if (Math.random() < 0.8) {
                const zone = availableZones[Math.floor(Math.random() * availableZones.length)];
                const pkg: Package = {
                    id: `pkg_${Date.now()}_${i}`,
                    pickupZone: zone,
                    destinationZone: this.state.deliveryZones[Math.floor(Math.random() * this.state.deliveryZones.length)],
                    status: "available",
                    points: BASE_PACKAGE_POINTS,
                    timeToLive: PACKAGE_TIME_TO_LIVE,
                    size: Math.random() < 0.7 ? "small" : Math.random() < 0.5 ? "medium" : "large",
                    color: "#8b4513"
                };
                zone.packages.push(pkg);
                this.state.packages.push(pkg);
            }
        }
        this.state.nextSpawnTime = Date.now() + SPAWN_INTERVAL;
        this.uiManager.frameRequested = true;
    }

    public endGame(): void {
        this.setInputLocked(true);
        this.state.gameStarted = false;
        this.calculateFinalScore();
        this.calculateRewards();
        this.showEndScreen();
        this.uiManager.frameRequested = true;
    }

    private calculateFinalScore(): void {
        this.state.timeBonus = Math.floor(this.state.timer * TIME_BONUS_PER_SECOND);
        this.state.score += this.state.timeBonus;
        this.state.score += this.state.perfectDeliveries * PERFECT_DELIVERY_BONUS;
        this.lastScore = this.state.score;
        this.lastPerfectDeliveries = this.state.perfectDeliveries;
    }

    private calculateRewards(): void {
        if (this.isPractice) {
            this.winnings = [];
            return;
        }
        const scoreMultiplier = rangeMapLinear(this.state.score, MIN_REWARD_AMOUNT, MAX_REWARD_AMOUNT, MIN_SCORE, MAX_SCORE, 1, 1);
        const baseFlunds = Math.floor(100 * scoreMultiplier);
        const baseResearch = Math.floor(10 * scoreMultiplier);
        this.winnings = [new Flunds(baseFlunds), new Resource("research", baseResearch)];
        progressMinigameOptionResearch(this.city, Math.floor(scoreMultiplier * 5));
    }

    private setInputLocked(locked: boolean): void {
        this.state.userInputLocked = locked;
    }

    private handleDragStart(pkg: Package, x: number, y: number): void {
        if (pkg.status !== "available" || this.state.userInputLocked) return;
        this.state.selectedPackage = pkg;
        pkg.status = "picked_up";
        const zoneDrawable = this.getZoneDrawable(pkg.pickupZone.id);
        if (zoneDrawable) {
            this.state.dragOffsetX = x - zoneDrawable.screenArea.x;
            this.state.dragOffsetY = y - zoneDrawable.screenArea.y;
        }
        this.uiManager.frameRequested = true;
    }

    private handleDragMove(x: number, y: number): void {
        if (!this.state.selectedPackage) return;
        this.state.highlightedZone = this.findHoveredZone(x, y);
        this.uiManager.frameRequested = true;
    }

    private handleDragEnd(x: number, y: number): void {
        if (!this.state.selectedPackage) return;
        const pkg = this.state.selectedPackage;
        const hoveredZone = this.state.highlightedZone;
        if (hoveredZone && hoveredZone.type === "vehicle" && hoveredZone.vehicle) {
            if (hoveredZone.vehicle.loadPackage(pkg)) {
                this.showFloatingText(hoveredZone, "+Load", "#44ff44");
            } else {
                this.state.vehicleFullMessageShown = true;
                this.showFloatingText(hoveredZone, "Full!", "#ff4444");
                pkg.status = "available";
            }
        } else {
            pkg.status = "available";
        }
        this.state.selectedPackage = null;
        this.state.highlightedZone = null;
        this.uiManager.frameRequested = true;
    }

    private findHoveredZone(x: number, y: number): LocationZone | null {
        const allZones = [...this.state.pickupZones, ...this.state.deliveryZones];
        for (const zone of allZones) {
            const zoneX = zone.gridX * TILE_SIZE;
            const zoneY = zone.gridY * TILE_SIZE;
            const zoneWidth = zone.width * TILE_SIZE;
            const zoneHeight = zone.height * TILE_SIZE;
            if (x >= zoneX && x <= zoneX + zoneWidth && y >= zoneY && y <= zoneY + zoneHeight) {
                return zone;
            }
        }
        for (const vehicle of this.state.vehicles) {
            const zone = vehicle.currentZone || vehicle.targetZone;
            if (!zone) continue;
            const zoneX = zone.gridX * TILE_SIZE;
            const zoneY = zone.gridY * TILE_SIZE;
            if (x >= zoneX && x <= zoneX + TILE_SIZE && y >= zoneY && y <= zoneY + TILE_SIZE) {
                return { ...zone, type: "vehicle" as const, vehicle: vehicle };
            }
        }
        return null;
    }

    private getZoneDrawable(zoneId: string): Drawable | undefined {
        return this.zoneDrawables.get(zoneId);
    }

    private showFloatingText(zone: LocationZone, text: string, color: string): void {
        const floatingText = new Drawable({
            x: zone.gridX * TILE_SIZE + ZONE_SIZE * TILE_SIZE / 2 - 50,
            y: zone.gridY * TILE_SIZE,
            width: "100px",
            height: "30px",
            text: text,
            fallbackColor: '#00000000'
        });
        if (this.lastDrawable) {
            this.lastDrawable.children.push(floatingText);
            const startY = floatingText.y;
            const duration = 1000;
            const startTime = Date.now();
            const animate = () => {
                const elapsed = Date.now() - startTime;
                const progress = elapsed / duration;
                floatingText.y = startY - progress * 50;
                if (progress < 1) {
                    requestAnimationFrame(animate);
                } else {
                    const index = this.lastDrawable!.children.indexOf(floatingText);
                    if (index > -1) this.lastDrawable!.children.splice(index, 1);
                }
                this.uiManager.frameRequested = true;
            };
            requestAnimationFrame(animate);
        }
    }

    public asDrawable(): Drawable {
        if (!this.shown) return this.lastDrawable = new Drawable({ width: "0px" });
        const mainDrawable = new Drawable({ width: "100%", height: "100%", fallbackColor: '#1a1a2e' });
        if (!this.state.gameStarted) {
            this.drawStartOverlay(mainDrawable);
        } else {
            this.drawGameArea(mainDrawable);
            this.drawHUD(mainDrawable);
        }
        this.lastDrawable = mainDrawable;
        return mainDrawable;
    }

    private drawStartOverlay(parent: Drawable): void {
        const overlay = parent.addChild(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            width: "min(100%, 600px)",
            height: "100%",
            fallbackColor: '#0a0a1a',
            onDrag: (x: number, y: number) => { this.scroller.handleDrag(y, overlay.screenArea); },
            onDragEnd: () => { this.scroller.resetDrag(); },
        }));
        if (this.state.howToPlayShown) {
            this.drawHowToPlay(overlay);
            return;
        }
        let nextY = 10 - this.scroller.getScroll();
        const baseY = nextY;
        overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "100%",
            height: "64px",
            text: "Parcel Patrol",
            biggerOnMobile: true,
            scaleYOnMobile: true
        }));
        nextY += 100;
        const startButton = overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "220px",
            height: "56px",
            fallbackColor: '#e94560',
            onClick: () => this.startGame(),
            children: [new Drawable({ anchors: ["centerX"], y: 8, width: "calc(100% - 10px)", height: "100%", text: "Start Delivery", centerOnOwnX: true })]
        }));
        const unaffordable = !this.city.hasResources(this.getCosts(), false);
        addResourceCosts(startButton, this.getCosts(), 86 - (this.spendResearch ? 28 : 0), 70, false, false, false, 48, 10, 32, undefined, undefined, unaffordable, this.city);
        nextY += 90;
        overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "300px",
            height: "48px",
            fallbackColor: '#00000000',
            onClick: () => { this.isPractice = !this.isPractice; },
            children: [new Drawable({ x: 5, width: "48px", height: "48px", image: new TextureInfo(64, 64, this.isPractice ? "ui/checked" : "ui/unchecked") }),
            new Drawable({ anchors: ["right"], rightAlign: true, x: 5, y: 10, width: "calc(100% - 60px)", height: "100%", text: "Practice Run (no rewards)" })]
        }));
        nextY += 60;
        overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "220px",
            height: "48px",
            fallbackColor: '#16213e',
            onClick: () => { this.toggleRules(); },
            children: [new Drawable({ anchors: ["centerX"], y: 5, width: "calc(100% - 10px)", height: "100%", text: "How to Play", centerOnOwnX: true })]
        }));
        nextY += 60;
        if (this.winnings?.length) {
            nextY = this.drawWinnings(overlay, nextY);
        }
        this.scroller.setChildrenSize(nextY - baseY);
    }

    private drawWinnings(parent: Drawable, startY: number): number {
        let nextY = startY + 20;
        parent.addChild(new Drawable({ anchors: ['centerX'], centerOnOwnX: true, y: nextY, width: "min(100%, 500px)", height: "48px", text: "Last Run Results" }));
        nextY += 60;
        const winningsArea = parent.addChild(new Drawable({ anchors: ['centerX'], centerOnOwnX: true, y: nextY, width: "min(100%, 500px)", height: "200px", fallbackColor: '#1a1a2e' }));
        winningsArea.addChild(new Drawable({ anchors: ['centerX'], centerOnOwnX: true, y: 10, width: "250px", height: "32px", text: `Final Score: ${this.lastScore}` }));
        winningsArea.addChild(new Drawable({ anchors: ['centerX'], centerOnOwnX: true, y: 50, width: "250px", height: "32px", text: `Perfect Deliveries: ${this.lastPerfectDeliveries}` }));
        winningsArea.addChild(new Drawable({ x: 107, y: 100, width: "100%", fallbackColor: '#00000000' }));
        addResourceCosts(winningsArea.children[winningsArea.children.length - 1], this.winnings, 0, 0, false, false, false, 48, 8, 24, 4);
        return nextY + 220;
    }

    private drawHowToPlay(overlay: Drawable): void {
        overlay.addChild(new Drawable({ anchors: ['top', 'right'], y: 10, x: -10, width: "100px", height: "40px", text: "Back", onClick: () => { this.toggleRules(); } }));
        let nextY = 60 - this.scroller.getScroll();
        overlay.addChild(new Drawable({ anchors: ['centerX'], centerOnOwnX: true, y: nextY, width: "min(100%, 500px)", height: "48px", text: "How to Play" }));
        nextY += 70;
        nextY = this.drawRuleSection(overlay, nextY, "Pick Up Packages", "Drag packages from green pickup zones to your delivery vehicles.");
        nextY = this.drawRuleSection(overlay, nextY, "Plan Your Route", "Vehicles can carry multiple packages. Drop off packages at red delivery zones.");
        nextY = this.drawRuleSection(overlay, nextY, "Watch the Clock", "Complete as many deliveries as possible before time runs out.");
        nextY = this.drawRuleSection(overlay, nextY, "Build Combos", "Chain deliveries together for score multipliers. Don't let combos break!");
        nextY = this.drawRuleSection(overlay, nextY, "Direct Delivery", "Advanced: Drop packages directly on delivery zones for bonus points.");
        this.scroller.setChildrenSize(nextY + 200);
    }

    private drawRuleSection(parent: Drawable, startY: number, title: string, description: string): number {
        let nextY = startY;
        parent.addChild(new Drawable({ x: 20, y: nextY, width: "calc(100% - 40px)", height: "32px", text: title, fallbackColor: '#e94560' }));
        nextY += 40;
        parent.addChild(new Drawable({ x: 20, y: nextY, width: "calc(100% - 40px)", height: "60px", wordWrap: true, keepParentWidth: true, text: description }));
        nextY += 80;
        return nextY;
    }

    private drawGameArea(parent: Drawable): void {
        this.drawZones(parent, 0);
        this.drawVehicles(parent);
        this.drawPackages(parent);
    }

    private drawZones(parent: Drawable, startY: number): void {
        this.state.pickupZones.forEach(zone => {
            const zoneDrawable = parent.addChild(new Drawable({
                x: zone.gridX * TILE_SIZE + ZONE_PADDING,
                y: startY + zone.gridY * TILE_SIZE + ZONE_PADDING,
                width: (TILE_SIZE * zone.width) - ZONE_PADDING * 2,
                height: (TILE_SIZE * zone.height) - ZONE_PADDING * 2,
                fallbackColor: '#2d4a3e',
                image: new TextureInfo(TILE_SIZE * zone.width, TILE_SIZE * zone.height, "minigame/parcelzone")
            }));
            this.zoneDrawables.set(zone.id, zoneDrawable);
        });
        this.state.deliveryZones.forEach(zone => {
            const zoneDrawable = parent.addChild(new Drawable({
                x: zone.gridX * TILE_SIZE + ZONE_PADDING,
                y: startY + zone.gridY * TILE_SIZE + ZONE_PADDING,
                width: (TILE_SIZE * zone.width) - ZONE_PADDING * 2,
                height: (TILE_SIZE * zone.height) - ZONE_PADDING * 2,
                fallbackColor: '#4a2d3e',
                image: new TextureInfo(TILE_SIZE * zone.width, TILE_SIZE * zone.height, "minigame/deliveryzone")
            }));
            this.zoneDrawables.set(zone.id, zoneDrawable);
        });
    }

    private drawVehicles(parent: Drawable): void {
        this.state.vehicles.forEach(vehicle => {
            const zone = vehicle.currentZone || vehicle.targetZone;
            if (!zone) return;
            const vehicleDrawable = parent.addChild(new Drawable({
                x: zone.gridX * TILE_SIZE + VEHICLE_OFFSET,
                y: zone.gridY * TILE_SIZE + VEHICLE_OFFSET,
                width: `${VEHICLE_SIZE}px`,
                height: `${VEHICLE_SIZE}px`,
                fallbackColor: '#e94560',
                image: new TextureInfo(VEHICLE_SIZE, VEHICLE_SIZE, vehicle.isMoving ? "minigame/deliveryvan_moving" : "minigame/deliveryvan")
            }));
            this.vehicleDrawables.set(vehicle.id, vehicleDrawable);
        });
    }

    private drawPackages(parent: Drawable): void {
        this.state.packages.forEach(pkg => {
            if (pkg.status === "delivered" || pkg.status === "expired") return;
            const sourceZone = pkg.pickupZone;
            const pkgDrawable = parent.addChild(new Drawable({
                x: sourceZone.gridX * TILE_SIZE + PACKAGE_OFFSET,
                y: sourceZone.gridY * TILE_SIZE + PACKAGE_OFFSET,
                width: `${PACKAGE_SIZE}px`,
                height: `${PACKAGE_SIZE}px`,
                fallbackColor: pkg.color,
                image: new TextureInfo(PACKAGE_SIZE, PACKAGE_SIZE, `minigame/package_${pkg.size}`)
            }));
            this.packageDrawables.set(pkg.id, pkgDrawable);
        });
    }

    private drawHUD(parent: Drawable): void {
        const hud = parent.addChild(new Drawable({ anchors: ['bottom'], y: -80, width: "100%", height: "80px", fallbackColor: '#0f3460' }));
        hud.addChild(new Drawable({ x: 20, y: 20, width: "120px", height: "40px", text: `Time: ${this.state.timer}s`, fallbackColor: this.state.timer < 10 ? '#ff4444' : '#44ff44', reddize: this.state.timer < 5 }));
        hud.addChild(new Drawable({ anchors: ['centerX'], centerOnOwnX: true, x: 0, y: 20, width: "200px", height: "40px", text: `Score: ${this.state.score}` }));
        if (this.state.comboMultiplier > 1) {
            hud.addChild(new Drawable({ anchors: ['right'], rightAlign: true, x: -20, y: 20, width: "150px", height: "40px", text: `Combo: x${this.state.comboMultiplier.toFixed(1)}`, fallbackColor: '#ffd700' }));
        }
    }

    private showCountdownNumber(step: string): void {
        // Countdown visualization
    }

    private showEndScreen(): void {
        // End screen visualization
    }

    private showInsufficientResourcesMessage(): void {
        // Resource error display
    }

    private toggleRules(): void {
        this.state.howToPlayShown = !this.state.howToPlayShown;
        if (this.state.howToPlayShown) {
            this.scroller.resetScroll();
        }
    }

    public async preloadImages(): Promise<void> {
        if (this.preloaded) return;
        const urls: { [key: string]: string } = {
            "minigame/parcelzone": "assets/minigame/parcelzone.png",
            "minigame/deliveryzone": "assets/minigame/deliveryzone.png",
            "minigame/deliveryvan": "assets/minigame/deliveryvan.png",
            "minigame/deliveryvan_moving": "assets/minigame/deliveryvan_moving.png",
            "minigame/package_small": "assets/minigame/package_small.png",
            "minigame/package_medium": "assets/minigame/package_medium.png",
            "minigame/package_large": "assets/minigame/package_large.png"
        };
        await this.uiManager.renderer.loadMoreSprites(this.city, urls);
        this.preloaded = true;
    }

    getLastDrawable(): Drawable | null {
        return this.lastDrawable;
    }

    public onResize(): void {
        this.scroller.onResize();
    }
}
