// Imports for core game logic, UI, resources, and utilities
import { City } from "../game/City.js";
import { CityFlags } from "../game/CityFlags.js";
import { Effect, EffectType } from "../game/GridType.js";
import { HappinessReward, ResearchReward } from "../game/EventTypes.js";
import { LONG_TICKS_PER_DAY } from "../game/FundamentalConstants.js";
import { Concrete, Electronics, Flunds, Lumber } from "../game/ResourceTypes.js";
import { Resource } from "../game/Resource.js";
import { Drawable } from "../ui/Drawable.js";
import { IHasDrawable } from "../ui/IHasDrawable.js";
import { IOnResizeEvent } from "../ui/IOnResizeEvent.js";
import { StandardScroller } from "../ui/StandardScroller.js";
import { TextureInfo } from "../ui/TextureInfo.js";
import { UIManager } from "../ui/UIManager.js";
import { addResourceCosts, humanizeFloor } from "../ui/UIUtil.js";
import { OnePracticeRun, rangeMapLinear } from "./MinigameUtil.js";

// --- Game Configuration Constants ---
const GRID_WIDTH = 6;
const GRID_HEIGHT = 6;
const TILE_SIZE = 64;
const HUB_TILE_X = 2;
const HUB_TILE_Y = 2;
const GAME_DURATION_SECONDS = 120;

// Visual paths for assets
const UTILITY_ICON_PATH = "ui/utility";

// --- State Interfaces ---
interface BuildingState {
    x: number;
    y: number;
    needType: 'power' | 'water' | 'data' | null;
    needTimer: number;
    maxNeedTimer: number;
}

interface TruckState {
    isActive: boolean;
    targetBuildingIndex: number;
    progress: number; // 0 to 1 for animation
}

// --- Difficulty Settings ---
type Difficulty = 'easy' | 'medium' | 'hard';

interface DifficultySettings {
    gameSpeed: number; // Speed of timers and truck movement
    needSpawnChance: number;
    maxNeedDuration: number;
    availableTrucks: number;
    playCost: number;
    rewardMultiplier: number;
}

const DIFFICULTY_SETTINGS: Record<Difficulty, DifficultySettings> = {
    easy: { gameSpeed: 0.8, needSpawnChance: 0.01, maxNeedDuration: 12, availableTrucks: 3, playCost: 100, rewardMultiplier: 1 },
    medium: { gameSpeed: 1, needSpawnChance: 0.015, maxNeedDuration: 9, availableTrucks: 2, playCost: 200, rewardMultiplier: 1.3 },
    hard: { gameSpeed: 1.3, needSpawnChance: 0.02, maxNeedDuration: 6, availableTrucks: 1, playCost: 300, rewardMultiplier: 1.6 }
};

// --- Main Minigame Class ---
export class UtilityJunctionPlays implements IHasDrawable, IOnResizeEvent {
    private uiManager: UIManager;
    private city: City;

    // UI and State Management
    public scroller = new StandardScroller(false, true);
    public shown = false;
    public lastDrawable: Drawable | null = null;
    public preloaded = false;

    // Game State
    public gameStarted = false;
    public isPractice = false;
    public userInputLocked = false;
    public howToPlayShown = false;
    public selectedDifficulty: Difficulty = 'medium';

    // Gameplay State
    private buildings: BuildingState[] = [];
    private trucks: TruckState[] = [];
    private score = 0;
    private gameTimer = GAME_DURATION_SECONDS;
    private needsMissed = 0;

    // End Game State
    public endReason = "";
    public winnings: Resource[] = [];

    // Timeouts
    private gameLoopTimeout: NodeJS.Timeout | null = null;

    constructor(city: City, uiManager: UIManager) {
        this.city = city;
        this.uiManager = uiManager;
    }

    // --- Lifecycle and Setup ---
    public async preloadImages(): Promise<void> {
        if (this.preloaded) return;
        const urls: { [key: string]: string } = {
            "minigame/utilitybuilding": "assets/minigame/utilitybuilding.png",
            "minigame/utilityhub": "assets/minigame/utilityhub.png",
            "minigame/utilitytruck": "assets/minigame/utilitytruck.png",
            "minigame/utilitybg": "assets/minigame/utilitybg.png",
            "utility/power": "assets/ui/utility_power.png",
            "utility/water": "assets/ui/utility_water.png",
            "utility/data": "assets/ui/utility_data.png",
        };
        await this.uiManager.renderer.loadMoreSprites(this.city, urls);
        this.preloaded = true;
    }

    public show(): void { this.shown = true; }

    public hide(): void {
        this.shown = false;
        if (this.gameLoopTimeout) clearTimeout(this.gameLoopTimeout);
    }

    public getLastDrawable(): Drawable | null { return this.lastDrawable; }

    // --- Core Game Logic ---
    public startGame(): void {
        const cost = this.getCosts();
        if (!this.city.checkAndSpendResources(cost)) {
            return;
        }
        this.city.updateLastUserActionTime();
        this.city.fullSave();
        this.initializeGame();
    }

    private initializeGame(): void {
        this.gameStarted = true;
        this.score = 0;
        this.gameTimer = GAME_DURATION_SECONDS;
        this.needsMissed = 0;
        this.buildings = [];
        this.trucks = [];

        // Create buildings
        for (let y = 0; y < GRID_HEIGHT; y++) {
            for (let x = 0; x < GRID_WIDTH; x++) {
                // Skip the hub location
                if (x === HUB_TILE_X && y === HUB_TILE_Y) continue;
                this.buildings.push({
                    x, y,
                    needType: null,
                    needTimer: 0,
                    maxNeedTimer: DIFFICULTY_SETTINGS[this.selectedDifficulty].maxNeedDuration
                });
            }
        }

        // Create trucks
        const settings = DIFFICULTY_SETTINGS[this.selectedDifficulty];
        for (let i = 0; i < settings.availableTrucks; i++) {
            this.trucks.push({ isActive: false, targetBuildingIndex: -1, progress: 0 });
        }

        this.startGameLoop();
        this.uiManager.frameRequested = true;
    }

    private startGameLoop(): void {
        const settings = DIFFICULTY_SETTINGS[this.selectedDifficulty];
        const loop = () => {
            if (!this.gameStarted) return;

            // Update game timer
            this.gameTimer--;
            if (this.gameTimer <= 0) {
                this.endReason = "Time's up!";
                this.endGame();
                return;
            }

            // Update building needs
            this.buildings.forEach(b => {
                if (b.needType) {
                    b.needTimer -= settings.gameSpeed;
                    if (b.needTimer <= 0) {
                        this.needsMissed++;
                        b.needType = null;
                    }
                }
            });

            // Spawn new needs
            if (Math.random() < settings.needSpawnChance) {
                const availableBuildings = this.buildings.filter(b => !b.needType);
                if (availableBuildings.length > 0) {
                    const building = availableBuildings[Math.floor(Math.random() * availableBuildings.length)];
                    const needs: BuildingState['needType'][] = ['power', 'water', 'data'];
                    building.needType = needs[Math.floor(Math.random() * needs.length)];
                    building.needTimer = building.maxNeedTimer;
                }
            }

            // Update trucks
            this.trucks.forEach(truck => {
                if (truck.isActive) {
                    truck.progress += 0.05 * settings.gameSpeed; // Truck speed
                    if (truck.progress >= 1) {
                        // Truck arrived
                        const building = this.buildings[truck.targetBuildingIndex];
                        if (building && building.needType) {
                            this.score += 10; // Base points for delivery
                            building.needType = null; // Need fulfilled
                        }
                        truck.isActive = false;
                        truck.targetBuildingIndex = -1;
                        truck.progress = 0;
                    }
                }
            });

            this.uiManager.frameRequested = true;
            this.gameLoopTimeout = setTimeout(loop, 100);
        };
        loop();
    }

    private endGame(): void {
        this.gameStarted = false;
        if (this.gameLoopTimeout) {
            clearTimeout(this.gameLoopTimeout);
            this.gameLoopTimeout = null;
        }
        this.calculateWinnings();
        this.uiManager.frameRequested = true;
    }

    private calculateWinnings(): void {
        if (this.isPractice) {
            this.winnings = [];
            return;
        }
        this.winnings = [];
        const settings = DIFFICULTY_SETTINGS[this.selectedDifficulty];
        const finalScore = this.score - (this.needsMissed * 5);

        const flundsAmount = rangeMapLinear(finalScore, 0, 250, 0, 50, 1, settings.rewardMultiplier);
        if (flundsAmount > 0) this.winnings.push(new Flunds(flundsAmount));

        const lumberAmount = rangeMapLinear(finalScore, 0, 250, 0, 5, 1, settings.rewardMultiplier);
        if (lumberAmount > 0) this.winnings.push(new Lumber(lumberAmount));

        const electronicsAmount = rangeMapLinear(finalScore, 0, 250, 0, 3, 1, settings.rewardMultiplier);
        if (electronicsAmount > 0) this.winnings.push(new Electronics(electronicsAmount));

        const concreteAmount = rangeMapLinear(finalScore, 0, 250, 0, 3, 1, settings.rewardMultiplier);
        if (concreteAmount > 0) this.winnings.push(new Concrete(concreteAmount));

        // Add a city-wide reward for high scores
        if (finalScore > 150) {
            this.city.events.push(new HappinessReward(LONG_TICKS_PER_DAY * 3, 0.05));
            this.city.events.push(new ResearchReward(LONG_TICKS_PER_DAY * 3, 5));
        }
    }

    public getCosts(): { type: string, amount: number }[] {
        if (this.isPractice) return OnePracticeRun;
        const cost = DIFFICULTY_SETTINGS[this.selectedDifficulty].playCost;
        return [{ type: "flunds", amount: cost }];
    }

    // --- User Interaction ---
    private toggleRules(): void {
        this.howToPlayShown = !this.howToPlayShown;
        if (this.howToPlayShown) this.scroller.resetScroll();
        this.uiManager.frameRequested = true;
    }

    private handleBuildingClick(buildingIndex: number): void {
        if (!this.gameStarted || this.userInputLocked) return;

        const building = this.buildings[buildingIndex];
        if (!building || !building.needType) return;

        // Find an available truck
        const availableTruck = this.trucks.find(t => !t.isActive);
        if (!availableTruck) {
            // Optional: Give user feedback that no trucks are available
            return;
        }

        // Dispatch truck
        availableTruck.isActive = true;
        availableTruck.targetBuildingIndex = buildingIndex;
        availableTruck.progress = 0;
        building.needType = null; // Immediately claim the need to prevent double-dispatch

        this.uiManager.frameRequested = true;
    }

    // --- UI Drawing ---
    public asDrawable(): Drawable {
        if (!this.shown) return this.lastDrawable = new Drawable({ width: "0px" });

        const mainDrawable = new Drawable({
            width: "100%",
            height: "100%",
            fallbackColor: '#222222'
        });

        if (!this.gameStarted) {
            this.drawStartOverlay(mainDrawable);
        } else {
            this.drawGameArea(mainDrawable);
        }

        this.lastDrawable = mainDrawable;
        return mainDrawable;
    }

    private drawStartOverlay(parent: Drawable): void {
        const overlay = parent.addChild(new Drawable({
            anchors: ["centerX"], centerOnOwnX: true, width: "min(100%, 600px)", height: "100%",
            fallbackColor: '#111111',
            onDrag: (x, y) => { this.scroller.handleDrag(y, overlay.screenArea!); },
            onDragEnd: () => { this.scroller.resetDrag(); },
        }));

        if (this.howToPlayShown) { this.drawHowToPlay(overlay); return; }

        let nextY = 10 - this.scroller.getScroll();
        let baseY = nextY;

        overlay.addChild(new Drawable({ anchors: ['centerX'], centerOnOwnX: true, y: nextY, width: "100%", text: "Utility Junction", height: "48px" }));
        nextY += 60;

        // Difficulty Selector
        const difficulties: { id: Difficulty, text: string }[] = [
            { id: 'easy', text: 'Easy' }, { id: 'medium', text: 'Medium' }, { id: 'hard', text: 'Hard' }
        ];
        const diffGroup = 'uj-d';
        const diffArea = overlay.addChild(new Drawable({ anchors: ['centerX'], centerOnOwnX: true, y: nextY, width: "400px", height: "60px", fallbackColor: '#00000000' }));
        difficulties.forEach((diff, i) => {
            diffArea.addChild(new Drawable({
                x: i * 130 + 5, y: 5, width: "120px", height: "50px", fallbackColor: '#444444',
                onClick: () => { this.selectedDifficulty = diff.id; this.uiManager.frameRequested = true; },
                children: [
                    new Drawable({ x: 5, y: 5, width: "40px", height: "40px", image: new TextureInfo(64, 64, this.selectedDifficulty === diff.id ? "ui/checked" : "ui/unchecked") }),
                    new Drawable({ x: 50, y: 7, width: "calc(100% - 60px)", height: "100%", text: diff.text })
                ]
            }));
        });
        this.city.minigameOptions.set(diffGroup, this.selectedDifficulty);
        nextY += 70;

        const startButton = overlay.addChild(new Drawable({ anchors: ['centerX'], centerOnOwnX: true, y: nextY, width: "220px", height: "48px", fallbackColor: '#444444', onClick: () => this.startGame() }));
        startButton.addChild(new Drawable({ anchors: ['centerX'], y: 5, width: "calc(100% - 10px)", height: "100%", text: "Start Game", centerOnOwnX: true }));
        const unaffordable = !this.city.hasResources(this.getCosts(), false);
        addResourceCosts(startButton, this.getCosts(), 86, 58, false, false, false, 48, 10, 32, undefined, undefined, unaffordable, this.city);
        nextY += 70;

        overlay.addChild(new Drawable({
            anchors: ['centerX'], centerOnOwnX: true, y: nextY, width: "300px", height: "48px", fallbackColor: '#00000000', onClick: () => { this.isPractice = !this.isPractice; this.uiManager.frameRequested = true; },
            children: [
                new Drawable({ x: 5, width: "48px", height: "48px", image: new TextureInfo(64, 64, this.isPractice ? "ui/checked" : "ui/unchecked") }),
                new Drawable({ anchors: ["right"], rightAlign: true, x: 5, y: 7, width: "calc(100% - 60px)", height: "100%", text: "Practice Run" })
            ]
        ]));
        nextY += 60;

        overlay.addChild(new Drawable({
            anchors: ['centerX'], centerOnOwnX: true, y: nextY, width: "220px", height: "48px", fallbackColor: '#444444', onClick: () => { this.toggleRules(); },
            children: [new Drawable({ anchors: ['centerX'], y: 5, width: "calc(100% - 10px)", height: "100%", text: "How to Play", centerOnOwnX: true })]
        }));
        nextY += 60;

        if (this.winnings.length > 0) {
            overlay.addChild(new Drawable({ anchors: ['centerX'], centerOnOwnX: true, y: nextY, width: "100%", height: "32px", text: "Last Rewards:" }));
            const rewardsArea = overlay.addChild(new Drawable({ anchors: ['centerX'], centerOnOwnX: true, y: nextY + 40, width: "calc(100% - 40px)", height: "100px", fallbackColor: '#00000000' }));
            addResourceCosts(rewardsArea, this.winnings, 0, 0, false, false, false, 64, 10, 32, 4);
            nextY += 150;
        }

        this.scroller.setChildrenSize(nextY - baseY);
    }

    private drawHowToPlay(parent: Drawable): void {
        let nextY = 10 - this.scroller.getScroll();
        const rules = [
            "Buildings on the grid will occasionally develop a need for Power, Water, or Data, shown by an icon and a timer.",
            "Click on a building with a need to dispatch a Utility Truck from the central hub.",
            "If the truck arrives before the timer expires, the need is fulfilled and you score points.",
            "If a timer runs out before a truck is dispatched, you miss an opportunity.",
            "Fulfill as many needs as possible before the game timer runs out to maximize your score and rewards.",
            "Higher difficulties increase the speed of the game, the frequency of needs, and reduce the number of available trucks."
        ];
        rules.forEach(rule => {
            parent.addChild(new Drawable({ anchors: ['centerX'], centerOnOwnX: true, y: nextY, width: "calc(100% - 40px)", height: "100%", text: rule, wordWrap: true, keepParentWidth: true }));
            nextY += 80;
        });
        parent.addChild(new Drawable({ anchors: ['centerX'], centerOnOwnX: true, y: nextY, width: "100%", height: "48px", text: "Click anywhere to close." }));
        parent.onClick = () => this.toggleRules();
        this.scroller.setChildrenSize(nextY + 60);
    }

    private drawGameArea(parent: Drawable): void {
        const gameArea = parent.addChild(new Drawable({ anchors: ['centerX'], centerOnOwnX: true, y: 80, width: (GRID_WIDTH * TILE_SIZE) + "px", height: (GRID_HEIGHT * TILE_SIZE) + "px", fallbackColor: '#333333' }));

        // Draw background grid
        for (let y = 0; y < GRID_HEIGHT; y++) {
            for (let x = 0; x < GRID_WIDTH; x++) {
                gameArea.addChild(new Drawable({ x: x * TILE_SIZE, y: y * TILE_SIZE, width: TILE_SIZE + "px", height: TILE_SIZE + "px", image: new TextureInfo(TILE_SIZE, TILE_SIZE, "minigame/utilitybg") }));
            }
        }

        // Draw buildings
        this.buildings.forEach((building, index) => {
            const buildingDrawable = gameArea.addChild(new Drawable({
                x: building.x * TILE_SIZE, y: building.y * TILE_SIZE, width: TILE_SIZE + "px", height: TILE_SIZE + "px",
                image: new TextureInfo(TILE_SIZE, TILE_SIZE, "minigame/utilitybuilding"),
                onClick: () => this.handleBuildingClick(index)
            }));
            if (building.needType) {
                buildingDrawable.addChild(new Drawable({ anchors: ['centerX', 'centerY'], centerOnOwnX: true, y: -10, width: "24px", height: "24px", image: new TextureInfo(24, 24, `utility/${building.needType}`) }));
                // Timer ring
                const timerRatio = building.needTimer / building.maxNeedTimer;
                buildingDrawable.addChild(new Drawable({
                    anchors: ['centerX', 'centerY'], centerOnOwnX: true, y: -10, width: "30px", height: "30px",
                    clipWidth: 0.1 + (1 - timerRatio) * 0.8, // Inverted clip to show depletion
                    image: new TextureInfo(30, 30, "ui/ring"), reddize: timerRatio < 0.25
                }));
            }
        });

        // Draw hub
        gameArea.addChild(new Drawable({ x: HUB_TILE_X * TILE_SIZE, y: HUB_TILE_Y * TILE_SIZE, width: TILE_SIZE + "px", height: TILE_SIZE + "px", image: new TextureInfo(TILE_SIZE, TILE_SIZE, "minigame/utilityhub") }));

        // Draw active trucks
        this.trucks.forEach(truck => {
            if (truck.isActive && truck.targetBuildingIndex >= 0) {
                const building = this.buildings[truck.targetBuildingIndex];
                if (!building) return;

                const startX = HUB_TILE_X * TILE_SIZE + TILE_SIZE / 2;
                const startY = HUB_TILE_Y * TILE_SIZE + TILE_SIZE / 2;
                const endX = building.x * TILE_SIZE + TILE_SIZE / 2;
                const endY = building.y * TILE_SIZE + TILE_SIZE / 2;

                const currentX = startX + (endX - startX) * truck.progress;
                const currentY = startY + (endY - startY) * truck.progress;

                gameArea.addChild(new Drawable({
                    x: currentX - TILE_SIZE / 2, y: currentY - TILE_SIZE / 2, width: TILE_SIZE + "px", height: TILE_SIZE + "px",
                    image: new TextureInfo(TILE_SIZE, TILE_SIZE, "minigame/utilitytruck")
                }));
            }
        });

        // Draw UI elements
        this.drawGameUI(parent);
    }

    private drawGameUI(parent: Drawable): void {
        const settings = DIFFICULTY_SETTINGS[this.selectedDifficulty];
        let nextY = 10;

        const statusBar = parent.addChild(new Drawable({ anchors: ['centerX'], centerOnOwnX: true, y: 10, width: "min(80%, 500px)", height: "60px", fallbackColor: '#444444' }));

        statusBar.addChild(new Drawable({ x: 10, y: 5, width: "150px", height: "50px", text: `Score: ${this.score}` }));
        statusBar.addChild(new Drawable({ anchors: ['right'], rightAlign: true, x: 10, y: 5, width: "150px", height: "50px", text: `Time: ${humanizeFloor(this.gameTimer)}s` }));

        const availableTrucks = this.trucks.filter(t => !t.isActive).length;
        statusBar.addChild(new Drawable({ anchors: ['centerX'], centerOnOwnX: true, y: 5, width: "150px", height: "50px", text: `Trucks: ${availableTrucks}/${settings.availableTrucks}` }));
    }

    public onResize(): void {
        this.scroller.onResize();
    }
}
