// GardenGrid.ts
import { TitleTypes } from "../game/AchievementTypes.js";
import { City } from "../game/City.js";
import { CityFlags } from "../game/CityFlags.js";
import { ProductionReward, HappinessReward } from "../game/EventTypes.js";
import { LONG_TICKS_PER_DAY } from "../game/FundamentalConstants.js";
import { UIManager } from "../ui/UIManager.js";
import { Resource } from "../game/Resource.js";
import { Flunds, Food, getResourceType } from "../game/ResourceTypes.js";
import { TextureInfo } from "../ui/TextureInfo.js";
import { Drawable } from "../ui/Drawable.js";
import { addResourceCosts, humanizeFloor } from "../ui/UIUtil.js";
import { IHasDrawable } from "../ui/IHasDrawable.js";
import { IOnResizeEvent } from "../ui/IOnResizeEvent.js";
import { StandardScroller } from "../ui/StandardScroller.js";
import { inPlaceShuffle } from "../game/MiscFunctions.js";
import { GameState } from "../game/GameState.js";
import { drawMinigameOptions } from "../ui/MinigameOptions.js";
import { OnePracticeRun, progressMinigameOptionResearch } from "./MinigameUtil.js";
import { EffectType } from "../game/GridType.js";

// Plant types with their properties
interface PlantType {
    id: string;
    name: string;
    icon: string;
    resources: { type: string, amount: number }[];
    baseAesthetic: number;
    productionBonus: number; // Bonus when adjacent to same type
    color: string;
}

interface GameState {
    grid: (PlantType | null)[][];
    score: number;
    resources: { type: string, amount: number }[];
    timeRemaining: number;
    moves: number;
    selectedPlant: PlantType | null;
    specialAbilities: { type: string, uses: number }[];
}

interface DifficultySettings {
    gridSize: number;
    timeLimit: number; // seconds, 0 for no time limit
    plantTypes: number;
    specialAbilityUses: number;
    rewardMultiplier: number;
}

const DIFFICULTY_SETTINGS: Record<string, DifficultySettings> = {
    casual: { gridSize: 5, timeLimit: 0, plantTypes: 4, specialAbilityUses: 3, rewardMultiplier: 1 },
    normal: { gridSize: 6, timeLimit: 120, plantTypes: 5, specialAbilityUses: 2, rewardMultiplier: 1.25 },
    expert: { gridSize: 7, timeLimit: 90, plantTypes: 6, specialAbilityUses: 1, rewardMultiplier: 1.5 }
};

const PLANT_TYPES: PlantType[] = [
    {
        id: "vegetable",
        name: "Vegetable Patch",
        icon: "minigame/garden/vegetable",
        resources: [{ type: "Food", amount: 3 }],
        baseAesthetic: 2,
        productionBonus: 1,
        color: "#4CAF50"
    },
    {
        id: "flower",
        name: "Flower Bed",
        icon: "minigame/garden/flower",
        resources: [{ type: "Happiness", amount: 2 }],
        baseAesthetic: 4,
        productionBonus: 0,
        color: "#E91E63"
    },
    {
        id: "fruit",
        name: "Fruit Tree",
        icon: "minigame/garden/fruit",
        resources: [{ type: "Food", amount: 4 }, { type: "Flunds", amount: 1 }],
        baseAesthetic: 3,
        productionBonus: 1,
        color: "#FF9800"
    },
    {
        id: "herb",
        name: "Herb Garden",
        icon: "minigame/garden/herb",
        resources: [{ type: "Pharmaceuticals", amount: 2 }],
        baseAesthetic: 3,
        productionBonus: 0,
        color: "#8BC34A"
    },
    {
        id: "berry",
        name: "Berry Bush",
        icon: "minigame/garden/berry",
        resources: [{ type: "Food", amount: 2 }, { type: "VitaminB12", amount: 1 }],
        baseAesthetic: 2,
        productionBonus: 1,
        color: "#9C27B0"
    },
    {
        id: "mushroom",
        name: "Mushroom Log",
        icon: "minigame/garden/mushroom",
        resources: [{ type: "Pharmaceuticals", amount: 3 }],
        baseAesthetic: 1,
        productionBonus: 2,
        color: "#795548"
    }
];

const SPECIAL_ABILITIES = [
    { id: "remove", name: "Remove Plant", icon: "ui/remove", description: "Remove a plant from the grid" },
    { id: "swap", name: "Swap Plants", icon: "ui/swap", description: "Swap two plants on the grid" },
    { id: "randomize", name: "Randomize Row", icon: "ui/shuffle", description: "Randomize a row of plants" }
];

export class GardenGrid implements IHasDrawable, IOnResizeEvent {
    private state: GameState;
    private uiManager: UIManager;
    private city: City;
    private difficulty: string = "normal";
    private settings: DifficultySettings;
    private timerTimeout: NodeJS.Timeout | null = null;
    private scroller: StandardScroller;
    private shown: boolean = false;
    private gameStarted: boolean = false;
    private lastDrawable: Drawable | null = null;
    private preloaded: boolean = false;
    private endReason: string = "";
    private winnings: Resource[] = [];
    private isPractice: boolean = false;

    constructor(city: City, uiManager: UIManager) {
        this.city = city;
        this.uiManager = uiManager;
        this.scroller = new StandardScroller(false, true);
        this.settings = DIFFICULTY_SETTINGS[this.difficulty];
        this.initializeState();
    }

    private initializeState(): void {
        const gridSize = this.settings.gridSize;
        this.state = {
            grid: Array(gridSize).fill(null).map(() => Array(gridSize).fill(null)),
            score: 0,
            resources: [],
            timeRemaining: this.settings.timeLimit,
            moves: 0,
            selectedPlant: null,
            specialAbilities: SPECIAL_ABILITIES.map(ability => ({
                type: ability.id,
                uses: this.settings.specialAbilityUses
            }))
        };
    }

    public show(): void {
        this.shown = true;
        this.preloadImages();
    }

    public hide(): void {
        this.shown = false;
        if (this.timerTimeout) {
            clearTimeout(this.timerTimeout);
            this.timerTimeout = null;
        }
    }

    public onResize(): void {
        this.scroller.onResize();
    }

    public async preloadImages(): Promise<void> {
        if (this.preloaded) return;

        const urls: { [key: string]: string } = {};

        // Plant icons
        PLANT_TYPES.forEach(plant => {
            urls[plant.icon] = `assets/${plant.icon}.png`;
        });

        // Ability icons
        SPECIAL_ABILITIES.forEach(ability => {
            urls[ability.icon] = `assets/${ability.icon}.png`;
        });

        // UI elements
        urls["minigame/garden/grid"] = "assets/minigame/garden/grid.png";
        urls["minigame/garden/selected"] = "assets/minigame/garden/selected.png";
        urls["minigame/garden/valid"] = "assets/minigame/garden/valid.png";
        urls["minigame/garden/invalid"] = "assets/minigame/garden/invalid.png";

        await this.uiManager.renderer.loadMoreSprites(this.city, urls);
        this.preloaded = true;
    }

    private getCosts(): { type: string, amount: number }[] {
        return this.isPractice ? OnePracticeRun :
            [{ type: "Flunds", amount: 500 }];
    }

    public startGame(): void {
        if (this.city.checkAndSpendResources(this.getCosts())) {
            this.initializeState();
            this.gameStarted = true;
            this.city.updateLastUserActionTime();
            this.city.fullSave();

            if (this.settings.timeLimit > 0) {
                this.startTimer();
            }
        }
    }

    private startTimer(): void {
        this.timerTimeout = setTimeout(() => {
            if (!this.gameStarted) return;

            this.state.timeRemaining--;
            if (this.state.timeRemaining <= 0) {
                this.endReason = "Time's up!";
                this.endGame();
            } else {
                this.startTimer();
            }
            this.uiManager.frameRequested = true;
        }, 1000);
    }

    private endGame(): void {
        this.gameStarted = false;
        if (this.timerTimeout) {
            clearTimeout(this.timerTimeout);
            this.timerTimeout = null;
        }

        this.calculateWinnings();
        this.uiManager.frameRequested = true;
    }

    private calculateWinnings(): void {
        // Base winnings from resources
        this.winnings = [];

        // Calculate resource production
        const resourceMap: { [key: string]: number } = {};
        this.state.grid.forEach(row => {
            row.forEach(cell => {
                if (cell) {
                    cell.resources.forEach(resource => {
                        resourceMap[resource.type] = (resourceMap[resource.type] || 0) +
                            resource.amount * this.settings.rewardMultiplier;
                    });
                }
            });
        });

        // Convert to resources
        Object.entries(resourceMap).forEach(([type, amount]) => {
            this.winnings.push(new (getResourceType(type) as any)(amount));
        });

        // Bonus for aesthetic score
        const aestheticScore = this.calculateAestheticScore();
        if (aestheticScore > 50) {
            this.winnings.push(new Flunds(aestheticScore * 10));
        }

        // Apply special rewards based on achievements
        if (this.state.moves < 20) {
            this.city.checkAndAwardAchievement("GardenGridEfficient");
            this.winnings.push(new Food(5));
        }

        if (this.state.timeRemaining > this.settings.timeLimit / 2) {
            this.city.checkAndAwardAchievement("GardenGridSpeedy");
            this.winnings.push(new Flunds(100));
        }

        // Filter and convert winnings
        this.winnings = filterConvertAwardWinnings(this.city, this.winnings);
    }

    private calculateAestheticScore(): number {
        let score = 0;
        const gridSize = this.settings.gridSize;

        // Base aesthetic value
        this.state.grid.forEach(row => {
            row.forEach(cell => {
                if (cell) {
                    score += cell.baseAesthetic;
                }
            });
        });

        // Penalty for adjacent same plants
        for (let y = 0; y < gridSize; y++) {
            for (let x = 0; x < gridSize; x++) {
                const plant = this.state.grid[y][x];
                if (plant) {
                    // Check adjacent plants
                    const neighbors = [
                        [y - 1, x], [y + 1, x], [y, x - 1], [y, x + 1]
                    ];

                    neighbors.forEach(([ny, nx]) => {
                        if (ny >= 0 && ny < gridSize && nx >= 0 && nx < gridSize) {
                            const neighbor = this.state.grid[ny][nx];
                            if (neighbor && neighbor.id === plant.id) {
                                score -= 1; // Penalty for adjacent same plants
                            }
                        }
                    });
                }
            }
        }

        return Math.max(0, score);
    }

    private canPlacePlant(x: number, y: number, plant: PlantType): boolean {
        // Check if position is empty
        if (this.state.grid[y][x] !== null) {
            return false;
        }

        // For now, allow any placement
        // Could add more complex rules here
        return true;
    }

    private placePlant(x: number, y: number, plant: PlantType): void {
        if (!this.canPlacePlant(x, y, plant)) return;

        this.state.grid[y][x] = plant;
        this.state.moves++;

        // Calculate production with bonuses
        let production = 0;
        const neighbors = [
            [y - 1, x], [y + 1, x], [y, x - 1], [y, x + 1]
        ];

        neighbors.forEach(([ny, nx]) => {
            if (ny >= 0 && ny < this.settings.gridSize &&
                nx >= 0 && nx < this.settings.gridSize) {
                const neighbor = this.state.grid[ny][nx];
                if (neighbor && neighbor.id === plant.id) {
                    production += plant.productionBonus;
                }
            }
        });

        // Update score
        this.state.score += plant.baseAesthetic + production;

        // Add resources to state
        plant.resources.forEach(resource => {
            const existing = this.state.resources.find(r => r.type === resource.type);
            if (existing) {
                existing.amount += resource.amount;
            } else {
                this.state.resources.push({ ...resource });
            }
        });

        // Check if grid is full
        if (this.isGridFull()) {
            this.endReason = "Garden complete!";
            this.endGame();
        }

        this.uiManager.frameRequested = true;
    }

    private isGridFull(): boolean {
        return this.state.grid.every(row =>
            row.every(cell => cell !== null)
        );
    }

    private useSpecialAbility(abilityId: string, ...args: any[]): void {
        const ability = this.state.specialAbilities.find(a => a.type === abilityId);
        if (!ability || ability.uses <= 0) return;

        ability.uses--;

        switch (abilityId) {
            case "remove":
                // Remove plant at position (args[0], args[1])
                if (args.length >= 2) {
                    const x = args[0];
                    const y = args[1];
                    const plant = this.state.grid[y][x];
                    if (plant) {
                        this.state.grid[y][x] = null;
                        // Adjust score and resources
                        this.state.score -= plant.baseAesthetic;
                        plant.resources.forEach(resource => {
                            const existing = this.state.resources.find(r => r.type === resource.type);
                            if (existing) {
                                existing.amount -= resource.amount;
                                if (existing.amount <= 0) {
                                    this.state.resources = this.state.resources.filter(r => r !== existing);
                                }
                            }
                        });
                    }
                }
                break;

            case "swap":
                // Swap plants at positions (args[0], args[1]) and (args[2], args[3])
                if (args.length >= 4) {
                    const x1 = args[0];
                    const y1 = args[1];
                    const x2 = args[2];
                    const y2 = args[3];

                    const temp = this.state.grid[y1][x1];
                    this.state.grid[y1][x1] = this.state.grid[y2][x2];
                    this.state.grid[y2][x2] = temp;
                }
                break;

            case "randomize":
                // Randomize row at position args[0]
                if (args.length >= 1) {
                    const y = args[0];
                    const row = this.state.grid[y];
                    const emptyCells = row.map((cell, x) => cell ? null : x).filter(x => x !== null);
                    const plantCells = row.map((cell, x) => cell ? x : null).filter(x => x !== null);

                    // Clear the row
                    row.fill(null);

                    // Place random plants in non-empty cells
                    plantCells.forEach(x => {
                        const availablePlants = PLANT_TYPES.filter(plant =>
                            row.some((cell, cx) => cell && cell.id === plant.id) ? false : true
                        );
                        if (availablePlants.length > 0) {
                            const randomPlant = availablePlants[Math.floor(Math.random() * availablePlants.length)];
                            row[x] = randomPlant;
                        }
                    });
                }
                break;
        }

        this.uiManager.frameRequested = true;
    }

    private drawStartOverlay(parent: Drawable): void {
        const overlay = parent.addChild(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            width: "min(100%, 600px)",
            height: "100%",
            fallbackColor: '#111111',
            onDrag: (x: number, y: number) => { this.scroller.handleDrag(y, overlay.screenArea); },
            onDragEnd: () => { this.scroller.resetDrag(); },
        }));

        if (this.howToPlayShown) {
            this.drawHowToPlay(overlay, parent);
            return;
        }

        let nextY = 10 - this.scroller.getScroll();
        let baseY = nextY;

        overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "100%",
            height: "48px",
            text: "Garden Grid",
        }));
        nextY += 134;

        const startButton = overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "220px",
            height: "48px",
            fallbackColor: '#4CAF50',
            onClick: () => this.startGame(),
            children: [
                new Drawable({
                    anchors: ["centerX"],
                    y: 5,
                    width: "calc(100% - 10px)",
                    height: "100%",
                    text: "Start Garden",
                    centerOnOwnX: true
                })
            ]
        }));

        const unaffordable = !this.city.hasResources(this.getCosts(), false);
        addResourceCosts(startButton, this.getCosts(), 86, 58, false, false, false, 48, 10, 32, undefined, undefined, unaffordable, this.city);
        nextY += 176;

        // Difficulty selector
        overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "500px",
            height: "48px",
            fallbackColor: '#00000000',
            text: "Difficulty:",
        }));
        nextY += 40;

        Object.entries(DIFFICULTY_SETTINGS).forEach(([key, settings]) => {
            const difficultyButton = overlay.addChild(new Drawable({
                anchors: ['centerX'],
                centerOnOwnX: true,
                y: nextY,
                width: "220px",
                height: "48px",
                fallbackColor: this.difficulty === key ? '#4CAF50' : '#444444',
                onClick: () => {
                    this.difficulty = key;
                    this.settings = settings;
                    this.initializeState();
                    this.uiManager.frameRequested = true;
                },
                children: [
                    new Drawable({
                        anchors: ["centerX"],
                        y: 5,
                        width: "calc(100% - 10px)",
                        height: "100%",
                        text: key.charAt(0).toUpperCase() + key.slice(1),
                        centerOnOwnX: true
                    })
                ]
            }));
            nextY += 60;
        });

        // Practice mode toggle
        overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "500px",
            height: "48px",
            fallbackColor: '#00000000',
            onClick: () => { this.isPractice = !this.isPractice; },
            children: [
                new Drawable({
                    x: 5,
                    width: "48px",
                    height: "48px",
                    image: new TextureInfo(64, 64, this.isPractice ? "ui/checked" : "ui/unchecked"),
                }),
                new Drawable({
                    anchors: ["right"],
                    rightAlign: true,
                    x: 5,
                    y: 7,
                    width: "calc(100% - 60px)",
                    height: "100%",
                    text: "Practice Run (no rewards)",
                }),
            ]
        }));
        nextY += 60;

        // How to play button
        overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "220px",
            height: "48px",
            fallbackColor: '#444444',
            onClick: () => { this.toggleRules(); },
            children: [
                new Drawable({
                    anchors: ["centerX"],
                    y: 5,
                    width: "calc(100% - 10px)",
                    height: "100%",
                    text: "How to Play",
                    centerOnOwnX: true
                })
            ]
        }));
        nextY += 60;

        if (this.winnings?.length) {
            // Draw the winnings from the last playthrough
            const winningsArea = overlay.addChild(new Drawable({
                anchors: ['centerX'],
                centerOnOwnX: true,
                y: nextY,
                width: "min(100%, 500px)",
                height: "500px",
                fallbackColor: '#444444',
                id: "winningsArea"
            }));

            winningsArea.addChild(new Drawable({
                anchors: ['centerX'],
                centerOnOwnX: true,
                biggerOnMobile: true,
                scaleYOnMobile: true,
                y: 10,
                width: "250px",
                height: "32px",
                text: "Garden Score: " + this.state.score,
            }));

            winningsArea.addChild(new Drawable({
                anchors: ['centerX'],
                centerOnOwnX: true,
                biggerOnMobile: true,
                scaleYOnMobile: true,
                y: 58,
                width: "250px",
                height: "32px",
                text: "Rewards attained:",
            }));

            const resourceContainer = winningsArea.addChild(new Drawable({
                x: 107,
                y: 100,
                width: "100%",
                fallbackColor: '#00000000',
                scaleYOnMobile: true
            }));

            addResourceCosts(resourceContainer, this.winnings, 0, 0, false, false, false, 64, 10, 32, 4);
            nextY += 510;
        }

        this.scroller.setChildrenSize(nextY - baseY);
    }

    private drawHowToPlay(overlay: Drawable, root: Drawable): void {
        let parent = overlay;
        parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: 10 - this.scroller.getScroll(),
            width: "100%",
            height: "48px",
            text: "How to Play Garden Grid",
        }));

        root.onClick = () => this.toggleRules();

        parent = parent.addChild(new Drawable({
            x: 20,
            y: 80 - this.scroller.getScroll(),
            width: "calc(100% - 40px)",
            height: "40px",
            wordWrap: true,
            keepParentWidth: true,
            text: "Create beautiful gardens by placing plants in a grid. Each plant type produces different resources and has different aesthetic values.",
        }));

        parent = parent.addChild(new Drawable({
            anchors: ['bottom'],
            y: -50,
            width: "calc(100% - 40px)",
            height: "40px",
            wordWrap: true,
            keepParentWidth: true,
            text: "Adjacent plants of the same type increase production but reduce aesthetic value. Balance resource production with garden beauty!",
        }));

        // Plant types
        parent = parent.addChild(new Drawable({
            anchors: ['bottom'],
            y: -100,
            width: "calc(100% - 40px)",
            height: "40px",
            wordWrap: true,
            keepParentWidth: true,
            text: "Plant Types:",
        }));

        PLANT_TYPES.forEach(plant => {
            parent = parent.addChild(new Drawable({
                anchors: ['bottom'],
                y: -150,
                width: "calc(100% - 40px)",
                height: "40px",
                wordWrap: true,
                keepParentWidth: true,
                text: `${plant.name}: Produces ${plant.resources.map(r => `${r.amount} ${r.type}`).join(', ')}. Base aesthetic: ${plant.baseAesthetic}.`,
            }));
        });

        // Special abilities
        parent = parent.addChild(new Drawable({
            anchors: ['bottom'],
            y: -200,
            width: "calc(100% - 40px)",
            height: "40px",
            wordWrap: true,
            keepParentWidth: true,
            text: "Special Abilities:",
        }));

        SPECIAL_ABILITIES.forEach(ability => {
            parent = parent.addChild(new Drawable({
                anchors: ['bottom'],
                y: -250,
                width: "calc(100% - 40px)",
                height: "40px",
                wordWrap: true,
                keepParentWidth: true,
                text: `${ability.name}: ${ability.description}. Limited uses per game.`,
            }));
        });

        this.scroller.setChildrenSize(1500); // Rough estimate
    }

    private toggleRules(): void {
        // This would be implemented to show/hide the rules
        // For simplicity, we'll assume this is handled elsewhere
    }

    private drawGameArea(parent: Drawable): void {
        const mainArea = parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: 10,
            width: "100%",
            height: "100%",
            fallbackColor: '#2E7D32',
        }));

        // Draw the grid
        const gridSize = this.settings.gridSize;
        const cellSize = Math.min(60, (500 - (gridSize - 1) * 10) / gridSize);
        const gridStartX = (600 - (gridSize * cellSize + (gridSize - 1) * 10)) / 2;
        const gridStartY = 100;

        // Draw grid cells
        for (let y = 0; y < gridSize; y++) {
            for (let x = 0; x < gridSize; x++) {
                const cellX = gridStartX + x * (cellSize + 10);
                const cellY = gridStartY + y * (cellSize + 10);

                const cell = mainArea.addChild(new Drawable({
                    x: cellX,
                    y: cellY,
                    width: cellSize + "px",
                    height: cellSize + "px",
                    fallbackColor: this.state.grid[y][x] ? this.state.grid[y][x]!.color : '#81C784',
                    image: new TextureInfo(cellSize, cellSize, "minigame/garden/grid"),
                    onClick: () => this.handleCellClick(x, y),
                }));

                // Draw plant if present
                if (this.state.grid[y][x]) {
                    cell.addChild(new Drawable({
                        width: "100%",
                        height: "100%",
                        image: new TextureInfo(cellSize, cellSize, this.state.grid[y][x]!.icon),
                    }));
                }
            }
        }

        // Draw plant selector
        const selectorArea = mainArea.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: gridStartY + gridSize * (cellSize + 10) + 30,
            width: "100%",
            height: "100px",
            fallbackColor: '#1B5E20',
        }));

        PLANT_TYPES.slice(0, this.settings.plantTypes).forEach((plant, index) => {
            selectorArea.addChild(new Drawable({
                x: index * (cellSize + 10),
                y: 0,
                width: cellSize + "px",
                height: cellSize + "px",
                fallbackColor: plant.color,
                image: new TextureInfo(cellSize, cellSize, plant.icon),
                onClick: () => this.handleSelectPlant(plant),
            }));
        });

        // Draw selected plant indicator
        if (this.state.selectedPlant) {
            const selectedIndex = PLANT_TYPES.findIndex(p => p.id === this.state.selectedPlant!.id);
            if (selectedIndex >= 0 && selectedIndex < this.settings.plantTypes) {
                selectorArea.addChild(new Drawable({
                    x: selectedIndex * (cellSize + 10),
                    y: 0,
                    width: cellSize + "px",
                    height: cellSize + "px",
                    image: new TextureInfo(cellSize, cellSize, "minigame/garden/selected"),
                }));
            }
        }

        // Draw special abilities
        const abilitiesArea = mainArea.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: gridStartY + gridSize * (cellSize + 10) + 150,
            width: "100%",
            height: "100px",
            fallbackColor: '#1B5E20',
        }));

        this.state.specialAbilities.forEach((ability, index) => {
            abilitiesArea.addChild(new Drawable({
                x: index * (cellSize + 10),
                y: 0,
                width: cellSize + "px",
                height: cellSize + "px",
                fallbackColor: ability.uses > 0 ? '#FFC107' : '#757575',
                image: new TextureInfo(cellSize, cellSize, SPECIAL_ABILITIES.find(a => a.id === ability.type)!.icon),
                onClick: () => this.handleSelectAbility(ability.type),
                children: [
                    new Drawable({
                        anchors: ['bottom'],
                        y: -5,
                        width: "100%",
                        height: "20px",
                        text: ability.uses.toString(),
                        wordWrap: true,
                        centerOnOwnX: true,
                    })
                ]
            }));
        });

        // Draw score and resources
        const infoArea = mainArea.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: 10,
            width: "500px",
            height: "80px",
            fallbackColor: '#1B5E20',
        }));

        infoArea.addChild(new Drawable({
            anchors: ['left'],
            x: 10,
            y: 10,
            width: "200px",
            height: "30px",
            text: "Score: " + this.state.score,
        }));

        if (this.settings.timeLimit > 0) {
            infoArea.addChild(new Drawable({
                anchors: ['right'],
                x: 10,
                y: 10,
                width: "200px",
                height: "30px",
                text: "Time: " + this.state.timeRemaining + "s",
            }));
        }

        infoArea.addChild(new Drawable({
            anchors: ['left'],
            x: 10,
            y: 45,
            width: "200px",
            height: "30px",
            text: "Moves: " + this.state.moves,
        }));

        // Draw current resources
        const resourceContainer = infoArea.addChild(new Drawable({
            anchors: ['right'],
            x: 10,
            y: 10,
            width: "200px",
            height: "65px",
        }));

        addResourceCosts(resourceContainer, this.state.resources, 0, 0, false, false, false, 24, 5, 16, 4);

        // Draw end game button
        mainArea.addChild(new Drawable({
            anchors: ['bottom'],
            y: -60,
            width: "150px",
            height: "40px",
            fallbackColor: '#D32F2F',
            onClick: () => {
                this.endReason = "Game ended early";
                this.endGame();
            },
            children: [
                new Drawable({
                    anchors: ["centerX"],
                    y: 5,
                    width: "calc(100% - 10px)",
                    height: "100%",
                    text: "End Game",
                    centerOnOwnX: true
                })
            ]
        }));
    }

    private handleCellClick(x: number, y: number): void {
        if (!this.gameStarted) return;

        // Check if we're using a special ability
        if (this.selectedAbility) {
            switch (this.selectedAbility) {
                case "remove":
                    this.useSpecialAbility("remove", x, y);
                    break;
                case "swap":
                    if (!this.swapStart) {
                        this.swapStart = { x, y };
                    } else {
                        this.useSpecialAbility("swap", this.swapStart.x, this.swapStart.y, x, y);
                        this.swapStart = null;
                    }
                    break;
            }
            this.selectedAbility = null;
            return;
        }

        // Normal plant placement
        if (this.state.selectedPlant) {
            this.placePlant(x, y, this.state.selectedPlant);
        }
    }

    private handleSelectPlant(plant: PlantType): void {
        if (!this.gameStarted) return;
        this.state.selectedPlant = plant;
        this.selectedAbility = null;
        this.uiManager.frameRequested = true;
    }

    private handleSelectAbility(abilityId: string): void {
        if (!this.gameStarted) return;
        this.selectedAbility = abilityId;
        this.state.selectedPlant = null;
        this.uiManager.frameRequested = true;
    }

    private selectedAbility: string | null = null;
    private swapStart: { x: number, y: number } | null = null;

    public asDrawable(): Drawable {
        if (!this.shown) return this.lastDrawable = new Drawable({ width: "0px" });

        const mainDrawable = new Drawable({
            width: "100%",
            height: "100%",
            fallbackColor: '#2E7D32',
        });

        if (!this.gameStarted) {
            this.drawStartOverlay(mainDrawable);
        } else {
            this.drawGameArea(mainDrawable);
        }

        this.lastDrawable = mainDrawable;
        return mainDrawable;
    }

    getLastDrawable(): Drawable | null {
        return this.lastDrawable;
    }
}