﻿// ---------- RecycleRush.ts ----------
import { City } from "../game/City.js";
import { UIManager } from "../ui/UIManager.js";
import { Resource } from "../game/Resource.js";
import { Flunds, getResourceType } from "../game/ResourceTypes.js";
import { TextureInfo } from "../ui/TextureInfo.js";
import { Drawable } from "../ui/Drawable.js";
import { addResourceCosts, humanizeFloor } from "../ui/UIUtil.js";
import { IHasDrawable } from "../ui/IHasDrawable.js";
import { IOnResizeEvent } from "../ui/IOnResizeEvent.js";
import { StandardScroller } from "../ui/StandardScroller.js";
import { OnePracticeRun, progressMinigameOptionResearch, rangeMapLinear } from "./MinigameUtil.js";
import { EffectType } from "../game/GridType.js";

/* -------------------------------------------------------------------------
   CONSTANTS – tweak these to change game balance or appearance
   ------------------------------------------------------------------------- */
const LANE_COUNT = 3;
const ITEM_SPAWN_INTERVAL_MS = 1200;
const ITEM_FALL_SPEED_PX_PER_TICK = 4; // pixels per UI frame
const GAME_DURATION_SECONDS = 45;
const ITEM_TYPES = ["paper", "plastic", "glass"] as const;
type ItemType = typeof ITEM_TYPES[number];

const ITEM_TEXTURE: Record<ItemType, string> = {
    paper: "ui/recycle_paper",
    plastic: "ui/recycle_plastic",
    glass: "ui/recycle_glass",
};

const BIN_TEXTURE: Record<ItemType, string> = {
    paper: "ui/bin_paper",
    plastic: "ui/bin_plastic",
    glass: "ui/bin_glass",
};

const SCORE_PER_CORRECT = 10;
const SCORE_PENALTY_WRONG = -5;
const TIME_BONUS_PER_CORRECT = 2; // seconds added

/* -------------------------------------------------------------------------
   STATE INTERFACES
   ------------------------------------------------------------------------- */
interface FallingItem {
    /** Type of the recyclable item */
    type: ItemType;
    /** Current Y‑position (pixels) relative to the top of the game area */
    y: number;
    /** Which lane (0‑based) the item is in */
    lane: number;
    /** Whether the item has already been tapped (prevents double‑tap) */
    handled: boolean;
}

interface RecycleRushState {
    /** All items currently falling */
    items: FallingItem[];
    /** Player’s score */
    score: number;
    /** Seconds remaining – UI timer counts down */
    timer: number;
    /** Whether the main game loop is active */
    gameStarted: boolean;
    /** Whether the UI is currently visible */
    shown: boolean;
    /** Whether the “how‑to‑play” overlay is visible */
    howToPlayShown: boolean;
    /** Practice mode flag – no rewards */
    isPractice: boolean;
    /** Timeout handles for the spawn loop and the per‑frame tick */
    spawnTimeout?: NodeJS.Timeout;
    tickTimeout?: NodeJS.Timeout;
    /** Last drawable returned from asDrawable (required by IHasDrawable) */
    lastDrawable?: Drawable;
}

/* -------------------------------------------------------------------------
   MAIN MINIGAME CLASS
   ------------------------------------------------------------------------- */
export class RecycleRush implements IHasDrawable, IOnResizeEvent {
    /* -------------------------- PUBLIC PROPERTIES -------------------------- */
    public readonly name = "Recycle Rush";
    public readonly abbreviation = "rr";

    /* -------------------------- PRIVATE FIELDS --------------------------- */
    private state: RecycleRushState;
    private scroller: StandardScroller;
    private uiManager: UIManager;
    private city: City;

    /* -------------------------- CONSTRUCTOR ----------------------------- */
    constructor(city: City, uiManager: UIManager) {
        this.city = city;
        this.uiManager = uiManager;

        this.state = {
            items: [],
            score: 0,
            timer: GAME_DURATION_SECONDS,
            gameStarted: false,
            shown: false,
            howToPlayShown: false,
            isPractice: false,
        };

        // vertical scrolling for the rules overlay
        this.scroller = new StandardScroller(false, true);
    }

    /* -------------------------- PUBLIC API ------------------------------ */
    /** Show the minigame – called by the UI manager */
    public show(): void {
        this.state.shown = true;
        this.state.howToPlayShown = false;
        this.state.isPractice = false;
        this.uiManager.frameRequested = true;
    }

    /** Hide the minigame – called when leaving the screen */
    public hide(): void {
        this.state.shown = false;
        this.endGame(); // clean any pending timeouts
        this.uiManager.frameRequested = true;
    }

    /** Return the most recent drawable (required by IHasDrawable) */
    public getLastDrawable(): Drawable | null {
        return this.state.lastDrawable ?? null;
    }

    /** Main UI entry point – builds the drawable hierarchy each frame */
    public asDrawable(): Drawable {
        if (!this.state.shown) {
            this.state.lastDrawable = new Drawable({ width: "0px" });
            return this.state.lastDrawable;
        }

        const root = new Drawable({
            width: "100%",
            height: "100%",
            fallbackColor: "#111",
        });

        if (!this.state.gameStarted) {
            this.drawStartOverlay(root);
            if (this.state.howToPlayShown) this.drawHowToPlay(root);
        } else {
            this.drawGameArea(root);
        }

        this.state.lastDrawable = root;
        return root;
    }

    /** Pre‑load all sprites used by this minigame */
    public async preloadImages(): Promise<void> {
        const urls: { [key: string]: string } = {
            "ui/recycle_bg": "assets/minigame/recycle_bg.png",
            "ui/recycle_paper": "assets/minigame/recycle_paper.png",
            "ui/recycle_plastic": "assets/minigame/recycle_plastic.png",
            "ui/recycle_glass": "assets/minigame/recycle_glass.png",
            "ui/bin_paper": "assets/minigame/bin_paper.png",
            "ui/bin_plastic": "assets/minigame/bin_plastic.png",
            "ui/bin_glass": "assets/minigame/bin_glass.png",
            "ui/checked": "assets/ui/checked.png",
            "ui/unchecked": "assets/ui/unchecked.png",
        };
        await this.uiManager.renderer.loadMoreSprites(this.city, urls);
    }

    /** Called by the UI manager when the window is resized */
    public onResize(): void {
        this.scroller.onResize();
        this.uiManager.frameRequested = true;
    }

    /* -------------------------- GAME LOGIC ----------------------------- */
    /** Start a new game – validates costs and initialises state */
    public startGame(): void {
        if (!this.city.checkAndSpendResources(this.getCosts(), false)) return;

        // Spend resources (or practice run)
        if (!this.state.isPractice) this.city.checkAndSpendResources(this.getCosts(), true);

        this.initializeGame();
        this.city.updateLastUserActionTime();
        this.uiManager.frameRequested = true;
    }

    /** Returns the resource cost for a new run */
    private getCosts(): { type: string; amount: number }[] {
        return this.state.isPractice
            ? OnePracticeRun
            : [{ type: getResourceType(Flunds), amount: 500 }];
    }

    /** Reset all mutable fields for a fresh run */
    private initializeGame(): void {
        // Cancel any lingering timeouts from a previous run
        this.clearTimeouts();

        this.state.items = [];
        this.state.score = 0;
        this.state.timer = GAME_DURATION_SECONDS;
        this.state.gameStarted = true;
        this.state.howToPlayShown = false;

        // Kick off the spawn loop and the per‑frame tick
        this.spawnItem();
        this.tick();
    }

    /** Main per‑frame tick – moves items, checks for loss, requests redraw */
    private tick(): void {
        if (!this.state.gameStarted) return;

        // Move items down
        for (const item of this.state.items) {
            item.y += ITEM_FALL_SPEED_PX_PER_TICK;
        }

        // Remove items that fell off the bottom (missed)
        const bottomY = this.getGameAreaHeight();
        const missed = this.state.items.filter(i => i.y > bottomY && !i.handled);
        for (const miss of missed) {
            this.state.score = Math.max(0, this.state.score + SCORE_PENALTY_WRONG);
        }
        this.state.items = this.state.items.filter(i => i.y <= bottomY);

        // Request UI redraw
        this.uiManager.frameRequested = true;

        // Schedule next tick (approx 60 fps)
        this.state.tickTimeout = setTimeout(() => this.tick(), 1000 / 60);
    }

    /** Spawn a new item in a random lane after a fixed interval */
    private spawnItem(): void {
        if (!this.state.gameStarted) return;

        const lane = Math.floor(Math.random() * LANE_COUNT);
        const type = ITEM_TYPES[Math.floor(Math.random() * ITEM_TYPES.length)];
        const newItem: FallingItem = { type, lane, y: -64, handled: false };
        this.state.items.push(newItem);

        // Schedule next spawn
        this.state.spawnTimeout = setTimeout(() => this.spawnItem(), ITEM_SPAWN_INTERVAL_MS);
    }

    /** Player tapped a bin – determine if any falling item matches */
    private handleBinTap(binType: ItemType, lane: number): void {
        if (!this.state.gameStarted) return;

        // Find the first unhandled item in this lane that is within the tappable zone
        const tapZoneTop = this.getGameAreaHeight() - 96; // 96 px from bottom is the bin area
        const candidate = this.state.items.find(
            i => i.lane === lane && !i.handled && i.y >= tapZoneTop
        );

        if (candidate) {
            candidate.handled = true;
            if (candidate.type === binType) {
                // Correct!
                this.state.score += SCORE_PER_CORRECT;
                this.state.timer = Math.min(
                    GAME_DURATION_SECONDS,
                    this.state.timer + TIME_BONUS_PER_CORRECT
                );
            } else {
                // Wrong bin
                this.state.score = Math.max(0, this.state.score + SCORE_PENALTY_WRONG);
                this.state.timer = Math.max(5, this.state.timer - 2);
            }
            // Remove the item after a short animation delay
            setTimeout(() => {
                this.state.items = this.state.items.filter(i => i !== candidate);
                this.uiManager.frameRequested = true;
            }, 200);
        }
    }

    /** End the game – stop timers, calculate rewards, show result overlay */
    private endGame(): void {
        this.clearTimeouts();
        this.state.gameStarted = false;

        // Calculate rewards only if not a practice run
        if (!this.state.isPractice) {
            const winnings = this.calculateRewards();
            this.city.events.push(...winnings);
        }

        // Show a simple “Game Over” overlay via UI redraw
        this.uiManager.frameRequested = true;
    }

    /** Compute the reward resources based on final score */
    private calculateRewards(): Resource[] {
        // Linear scaling – you can replace with rangeMapLinear for more exotic curves
        const flunds = Math.round(rangeMapLinear(this.state.score, 0, 2000, 0, 1000, 1, 1));
        return [new Flunds(flunds)];
    }

    /** Cancel any active timeouts (spawn / tick) */
    private clearTimeouts(): void {
        if (this.state.spawnTimeout) clearTimeout(this.state.spawnTimeout);
        if (this.state.tickTimeout) clearTimeout(this.state.tickTimeout);
        this.state.spawnTimeout = undefined;
        this.state.tickTimeout = undefined;
    }

    /* -------------------------- UI DRAWING ----------------------------- */
    /** Draw the start screen – title, start button, practice toggle, rules button */
    private drawStartOverlay(parent: Drawable): void {
        const overlay = parent.addChild(
            new Drawable({
                anchors: ["centerX"],
                centerOnOwnX: true,
                width: "min(100%, 600px)",
                height: "100%",
                fallbackColor: "#222",
                id: "startOverlay",
                // Drag handling for the rules scroller (if rules are open)
                onDrag: (x, y) => {
                    if (this.state.howToPlayShown) this.scroller.handleDrag(y, overlay.screenArea);
                },
                onDragEnd: () => {
                    if (this.state.howToPlayShown) this.scroller.resetDrag();
                },
            })
        );

        let nextY = 30 - this.scroller.getScroll();

        // Game title
        overlay.addChild(
            new Drawable({
                anchors: ["centerX"],
                centerOnOwnX: true,
                y: nextY,
                width: "100%",
                height: "48px",
                text: "Recycle Rush",
                biggerOnMobile: true,
                scaleYOnMobile: true,
            })
        );
        nextY += 80;

        // Start button
        const startBtn = overlay.addChild(
            new Drawable({
                anchors: ["centerX"],
                centerOnOwnX: true,
                y: nextY,
                width: "220px",
                height: "48px",
                fallbackColor: "#006600",
                onClick: () => this.startGame(),
                children: [
                    new Drawable({
                        anchors: ["centerX"],
                        y: 5,
                        width: "calc(100% - 10px)",
                        height: "100%",
                        text: "Start Game",
                        centerOnOwnX: true,
                    }),
                ],
            })
        );

        // Show the cost next to the button
        const unaffordable = !this.city.hasResources(this.getCosts(), false);
        addResourceCosts(
            startBtn,
            this.getCosts(),
            20,
            20,
            false,
            false,
            false,
            48,
            8,
            32,
            undefined,
            undefined,
            unaffordable,
            this.city
        );

        nextY += 100;

        // Practice mode toggle
        overlay.addChild(
            new Drawable({
                anchors: ["centerX"],
                centerOnOwnX: true,
                y: nextY,
                width: "260px",
                height: "48px",
                fallbackColor: "#333",
                onClick: () => {
                    this.state.isPractice = !this.state.isPractice;
                    this.uiManager.frameRequested = true;
                },
                children: [
                    new Drawable({
                        x: 5,
                        width: "48px",
                        height: "48px",
                        image: new TextureInfo(64, 64, this.state.isPractice ? "ui/checked" : "ui/unchecked"),
                    }),
                    new Drawable({
                        anchors: ["right"],
                        rightAlign: true,
                        x: 5,
                        y: 7,
                        width: "calc(100% - 60px)",
                        height: "100%",
                        text: "Practice Run (no rewards)",
                    }),
                ],
            })
        );
        nextY += 70;

        // How‑to‑play button
        overlay.addChild(
            new Drawable({
                anchors: ["centerX"],
                centerOnOwnX: true,
                y: nextY,
                width: "220px",
                height: "48px",
                fallbackColor: "#004466",
                onClick: () => this.toggleRules(),
                children: [
                    new Drawable({
                        anchors: ["centerX"],
                        y: 5,
                        width: "calc(100% - 10px)",
                        height: "100%",
                        text: "How to Play",
                        centerOnOwnX: true,
                    }),
                ],
            })
        );

        // Update scroller size (no scrollable content on the start screen)
        this.scroller.setChildrenSize(0);
    }

    /** Show the rules overlay – scrollable text */
    private drawHowToPlay(root: Drawable): void {
        const overlay = root.addChild(
            new Drawable({
                anchors: ["centerX"],
                centerOnOwnX: true,
                width: "min(100%, 600px)",
                height: "100%",
                fallbackColor: "#111",
                id: "rulesOverlay",
                onDrag: (x, y) => this.scroller.handleDrag(y, overlay.screenArea),
                onDragEnd: () => this.scroller.resetDrag(),
            })
        );

        // Close button (top‑right)
        overlay.addChild(
            new Drawable({
                x: -10,
                y: 10,
                width: "48px",
                height: "48px",
                image: new TextureInfo(64, 64, "ui/close"),
                onClick: () => this.toggleRules(),
            })
        );

        let nextY = 70 - this.scroller.getScroll();

        const addParagraph = (text: string) => {
            overlay.addChild(
                new Drawable({
                    x: 20,
                    y: nextY,
                    width: "calc(100% - 40px)",
                    height: "40px",
                    wordWrap: true,
                    keepParentWidth: true,
                    text,
                })
            );
            nextY += 60;
        };

        addParagraph(
            "Items fall down three conveyor lanes. Tap the correct recycling bin before the item reaches the bottom."
        );
        addParagraph(
            "Correct matches give points and add a few seconds to the timer. Wrong matches deduct points and time."
        );
        addParagraph(
            "The game ends when the timer runs out. Your final score determines the Flunds reward."
        );
        addParagraph(
            "Practice runs let you try the game without spending any Flunds, but you won’t earn a reward."
        );

        // Set the scrollable height (rough estimate)
        this.scroller.setChildrenSize(nextY + 30);
    }

    /** Main gameplay UI – background, lanes, falling items, bins, score/timer */
    private drawGameArea(parent: Drawable): void {
        const area = parent.addChild(
            new Drawable({
                width: "100%",
                height: "100%",
                fallbackColor: "#003300",
                // Background image (optional)
                image: new TextureInfo(800, 600, "ui/recycle_bg"),
            })
        );

        // Draw lanes (simple vertical lines)
        const laneWidth = this.getLaneWidth();
        for (let i = 0; i < LANE_COUNT; i++) {
            area.addChild(
                new Drawable({
                    x: i * laneWidth + laneWidth / 2 - 1,
                    y: 0,
                    width: "2px",
                    height: "100%",
                    fallbackColor: "#555",
                })
            );
        }

        // Draw falling items
        for (const item of this.state.items) {
            area.addChild(
                new Drawable({
                    x: item.lane * laneWidth + laneWidth / 2 - 32,
                    y: item.y,
                    width: "64px",
                    height: "64px",
                    image: new TextureInfo(64, 64, ITEM_TEXTURE[item.type]),
                })
            );
        }

        // Draw bins at the bottom – each bin is clickable
        const binY = this.getGameAreaHeight() - 80;
        for (let lane = 0; lane < LANE_COUNT; lane++) {
            const binType = ITEM_TYPES[lane] as ItemType; // tie each lane to a type for simplicity
            area.addChild(
                new Drawable({
                    x: lane * laneWidth + laneWidth / 2 - 40,
                    y: binY,
                    width: "80px",
                    height: "80px",
                    image: new TextureInfo(80, 80, BIN_TEXTURE[binType]),
                    onClick: () => this.handleBinTap(binType, lane),
                })
            );
        }

        // Draw score & timer UI (top bar)
        this.drawResources(area);
        this.drawTimer(area);
    }

    /** Render the score and Flunds count */
    private drawResources(parent: Drawable): void {
        // Score
        parent.addChild(
            new Drawable({
                x: 20,
                y: 10,
                width: "200px",
                height: "32px",
                text: `Score: ${this.state.score}`,
                rightAlign: false,
            })
        );

        // Flunds (city currency)
        const flunds = this.city.getResourceAmount(getResourceType(Flunds));
        parent.addChild(
            new Drawable({
                x: -220,
                y: 10,
                width: "200px",
                height: "32px",
                text: `Flunds: ${humanizeFloor(flunds)}`,
                anchors: ["right"],
                rightAlign: true,
            })
        );
    }

    /** Render the countdown timer */
    private drawTimer(parent: Drawable): void {
        const timerY = 10;
        const timerWidth = 150;
        const timerHeight = 30;

        const timerPct = Math.max(0, this.state.timer) / GAME_DURATION_SECONDS;

        parent.addChild(
            new Drawable({
                anchors: ["centerX"],
                centerOnOwnX: true,
                y: timerY,
                width: `${timerWidth}px`,
                height: `${timerHeight}px`,
                fallbackColor: "#444",
                children: [
                    new Drawable({
                        clipWidth: timerPct,
                        width: "100%",
                        height: "100%",
                        fallbackColor: timerPct < 0.2 ? "#ff4444" : "#44ff44",
                    }),
                    new Drawable({
                        anchors: ["centerX"],
                        centerOnOwnX: true,
                        y: 4,
                        width: "100%",
                        height: "100%",
                        text: `${Math.ceil(this.state.timer)}s`,
                        rightAlign: false,
                    }),
                ],
            })
        );
    }

    /** Toggle the “how‑to‑play” overlay */
    private toggleRules(): void {
        this.state.howToPlayShown = !this.state.howToPlayShown;
        if (this.state.howToPlayShown) this.scroller.resetScroll();
        this.uiManager.frameRequested = true;
    }

    /* -------------------------- HELPERS ------------------------------- */
    /** Height of the interactive game area (excluding UI chrome) */
    private getGameAreaHeight(): number {
        // For simplicity we assume the full screen height minus a 100 px top UI bar
        // In a real implementation you would compute this from the parent drawable size.
        return window.innerHeight - 120;
    }

    /** Width of a single lane (based on screen width) */
    private getLaneWidth(): number {
        return Math.floor(window.innerWidth / LANE_COUNT);
    }
}
