import { City } from "../game/City.js";
import { UIManager } from "../ui/UIManager.js";
import { Resource } from "../game/Resource.js";
import { Concrete, Wood, Flunds } from "../game/ResourceTypes.js";
import { TextureInfo } from "../ui/TextureInfo.js";
import { Drawable } from "../ui/Drawable.js";
import { StandardScroller } from "../ui/StandardScroller.js";
import { IHasDrawable } from "../ui/IHasDrawable.js";
import { IOnResizeEvent } from "../ui/IOnResizeEvent.js";
import { addResourceCosts, humanizeFloor, longTicksToHoursAndMinutes } from "../ui/UIUtil.js";
import { OnePracticeRun } from "./MinigameUtil.js";
import { GameState } from "../game/GameState.js";

// --- Constants & Configuration ---
const TILE_SIZE = 48; // Pixel size of one grid cell
const GRID_COLS = 8;
const GRID_ROWS = 10;
const GAME_DURATION = 60; // Seconds to survive (or score based)
const COST_CONCRETE_PER_BLOCK = 15;
const REFUND_CONCRETE_PER_BLOCK = 7; // 50% refund
const STARTING_CONCRETE = 150;

// --- State Interface ---
interface LeveeGameState {
    grid: string[]; // Array of strings "x,y" representing placed blocks
    waterLevel: number; // 0.0 to 1.0 (bottom to top)
    concrete: number;
    score: number;
    gameOver: boolean;
    gameStarted: boolean;
    isPractice: boolean;
    endReason: string;
    showRules: boolean;
    timer: number;
}

// --- Main Minigame Class ---
export class LeveeMaster implements IHasDrawable, IOnResizeEvent {
    private state: LeveeGameState;
    private uiManager: UIManager;
    private timerTimeout: NodeJS.Timeout | null = null;
    private scroller: StandardScroller;
    private lastDrawable: Drawable | null = null;
    private preloaded: boolean = false;
    private shown: boolean = false;

    constructor(private city: City, private uiManager: UIManager) {
        this.state = {
            grid: [],
            waterLevel: 0.0,
            concrete: STARTING_CONCRETE,
            score: 0,
            gameOver: false,
            gameStarted: false,
            isPractice: false,
            endReason: "",
            showRules: false,
            timer: GAME_DURATION
        };
        this.scroller = new StandardScroller(false, false); // Vertical scroller
    }

    // --- Lifecycle Methods ---

    public async preloadImages(): Promise<void> {
        if (this.preloaded) return;
        const urls: { [key: string]: string } = {
            "minigame/levee_bg": "assets/minigame/levee_bg.png", // Background color handled by fallbackColor
            "minigame/concrete": "assets/minigame/concrete.png",
            "minigame/water": "assets/minigame/water.png",
            "minigame/citycenter": "assets/minigame/citycenter.png"
        };
        await this.uiManager.renderer.loadMoreSprites(this.city, urls);
        this.preloaded = true;
    }

    public show(): void {
        this.shown = true;
        if (!this.state.gameStarted) {
            this.uiManager.frameRequested = true;
        }
    }

    public hide(): void {
        this.shown = false;
        this.stopTimer();
        this.uiManager.frameRequested = true;
    }

    public onResize(): void {
        this.scroller.onResize();
    }

    public getLastDrawable(): Drawable | null {
        return this.lastDrawable;
    }

    // --- Core Logic Methods ---

    public startGame(): void {
        // Check costs
        const costs = this.getCosts();
        if (!this.city.checkAndSpendResources(costs)) {
            return; // Not enough resources
        }

        // Reset State
        this.state = {
            grid: [],
            waterLevel: 0.0,
            concrete: STARTING_CONCRETE,
            score: 0,
            gameOver: false,
            gameStarted: true,
            isPractice: this.state.isPractice,
            endReason: "",
            showRules: false,
            timer: GAME_DURATION
        };
        
        this.scroller.resetScroll();
        this.city.updateLastUserActionTime();
        this.city.fullSave();
        
        this.startTimer();
        this.uiManager.frameRequested = true;
    }

    private getCosts(): { type: string, amount: number }[] {
        // If practice, cost is just "OnePracticeRun" (essentially free or handled by city flags)
        // If real game, cost Flunds
        if (this.state.isPractice) {
            return OnePracticeRun;
        }
        return [{ type: "flunds", amount: 500 }];
    }

    private startTimer(): void {
        this.stopTimer();
        this.timerTimeout = setTimeout(() => {
            if (!this.state.gameStarted || this.state.gameOver) return;

            this.state.timer--;

            // Update Water Level
            this.state.waterLevel += 0.001; // Slowly rising water

            // Check Win/Loss
            if (this.state.waterLevel >= 1.0) {
                this.state.endReason = "The City Center Flooded!";
                this.endGame();
            } else if (this.state.timer <= 0) {
                this.state.endReason = "Time's Up!";
                this.endGame();
            } else {
                this.startTimer();
            }
            this.uiManager.frameRequested = true;
        }, 1000); // 1 second tick
    }

    private stopTimer(): void {
        if (this.timerTimeout) {
            clearTimeout(this.timerTimeout);
            this.timerTimeout = null;
        }
    }

    private handleTileClick(col: number, row: number): void {
        if (!this.state.gameStarted || this.state.gameOver) return;

        const key = `${ col },${ row } `;

        // If block exists, remove it (Refund)
        if (this.state.grid.includes(key)) {
            this.state.grid = this.state.grid.filter(k => k !== key);
            this.state.concrete += REFUND_CONCRETE_PER_BLOCK;
            this.state.score = Math.max(0, this.state.score - 1); // Lose score for removal? Or just keep it static? Let's keep score static to encourage keeping blocks.
        } else {
            // Try to add block
            if (this.state.concrete >= COST_CONCRETE_PER_BLOCK) {
                this.state.grid.push(key);
                this.state.concrete -= COST_CONCRETE_PER_BLOCK;
                this.state.score++;
            }
        }
        this.uiManager.frameRequested = true;
    }

    public toggleRules(): void {
        this.state.showRules = !this.state.showRules;
        if (this.state.showRules) {
            this.scroller.resetScroll();
        }
        this.uiManager.frameRequested = true;
    }

    private endGame(): void {
        this.state.gameOver = true;
        this.stopTimer();
        this.calculateRewards();
    }

    private calculateRewards(): void {
        if (this.state.isPractice) return;

        // Calculate winnings based on score
        // Base reward: 10 Concrete
        // Bonus: 1 Concrete per block placed
        const concreteWinnings = 10 + this.state.score;
        const woodWinnings = this.state.score / 2; // Half as much wood

        // Convert to Resource objects
        const rewards: Resource[] = [];
        
        // Add Concrete
        if (concreteWinnings > 0) rewards.push(new Concrete(concreteWinnings));
        // Add Wood (if any)
        if (woodWinnings > 0) rewards.push(new Wood(Math.floor(woodWinnings)));

        this.state.winnings = rewards;
    }

    // --- Drawing Methods ---

    public asDrawable(): Drawable {
        if (!this.shown) return this.lastDrawable = new Drawable({ width: "0px" });

        const root = new Drawable({
            width: "100%",
            height: "100%",
            fallbackColor: '#222222', // Dark background
            onDrag: (x: number, y: number) => { this.scroller.handleDrag(y, root.screenArea); },
            onDragEnd: () => { this.scroller.resetDrag(); },
        });

        // Main content wrapper for scrolling
        const contentWrapper = root.addChild(new Drawable({
            width: "100%",
            height: "100%",
            fallbackColor: '#111111',
            children: [
                new Drawable({
                    anchors: ["centerX"],
                    centerOnOwnX: true,
                    width: "min(100%, 600px)",
                    height: "100%",
                    children: [] // This is where we draw the game
                })
            ]
        }));

        if (!this.state.gameStarted) {
            this.drawStartScreen(contentWrapper.children[0] as Drawable);
        } else {
            // Draw Game Area
            this.drawGameArea(contentWrapper.children[0] as Drawable);
            
            // Draw End Overlay (if game over)
            if (this.state.gameOver) {
                this.drawEndOverlay(contentWrapper.children[0] as Drawable);
            }
        }

        // Draw Rules Overlay (if shown)
        if (this.state.showRules) {
            this.drawRulesOverlay(root, contentWrapper.children[0] as Drawable);
        }

        this.lastDrawable = root;
        return root;
    }

    private drawStartScreen(parent: Drawable): void {
        let nextY = 10 - this.scroller.getScroll();
        const baseY = nextY;

        // Title
        parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "100%",
            height: "48px",
            text: "Levee Master",
        }));
        nextY += 100;

        // Start Button
        const startButton = parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "220px",
            height: "48px",
            fallbackColor: '#444444',
            onClick: () => this.startGame(),
            children: [
                new Drawable({
                    anchors: ["centerX"],
                    y: 5,
                    width: "calc(100% - 10px)",
                    height: "100%",
                    text: "Start Game",
                    centerOnOwnX: true
                })
            ]
        }));

        // Cost Display
        const costs = this.getCosts();
        const canAfford = this.city.hasResources(costs, false);
        addResourceCosts(startButton, costs, 86, 58, false, false, false, 48, 10, 32, undefined, undefined, !canAfford, this.city);
        nextY += 120;

        // Practice Toggle
        parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "500px",
            height: "48px",
            fallbackColor: '#00000000',
            onClick: () => { this.state.isPractice = !this.state.isPractice; },
            children: [
                new Drawable({
                    x: 5,
                    width: "48px",
                    height: "48px",
                    image: new TextureInfo(64, 64, this.state.isPractice ? "ui/checked" : "ui/unchecked"),
                }),
                new Drawable({
                    anchors: ["right"],
                    rightAlign: true,
                    x: 5,
                    y: 7,
                    width: "calc(100% - 60px)",
                    height: "100%",
                    text: "Practice Run (Free)",
                }),
            ]
        }));
        nextY += 60;

        // Rules Button
        parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "220px",
            height: "48px",
            fallbackColor: '#444444',
            onClick: () => { this.toggleRules(); },
            children: [
                new Drawable({
                    anchors: ["centerX"],
                    y: 5,
                    width: "calc(100% - 10px)",
                    height: "100%",
                    text: "How to Play",
                    centerOnOwnX: true
                })
            ]
        }));
        nextY += 60;

        this.scroller.setChildrenSize(nextY - baseY);
    }

    private drawGameArea(parent: Drawable): void {
        // Calculate Layout
        const totalWidth = GRID_COLS * TILE_SIZE;
        const totalHeight = GRID_ROWS * TILE_SIZE;
        
        const gameContainer = parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: 50 - this.scroller.getScroll(),
            width: `${ totalWidth } px`,
            height: `${ totalHeight } px`,
            fallbackColor: '#00000000',
            children: [
                // Grid Background
                new Drawable({
                    anchors: ['left', 'top'],
                    x: 0, y: 0,
                    width: "100%", height: "100%",
                    fallbackColor: '#333333'
                })
            ]
        }));

        // Draw Grid
        for (let r = 0; r < GRID_ROWS; r++) {
            for (let c = 0; c < GRID_COLS; c++) {
                const tile = gameContainer.children[0].addChild(new Drawable({
                    anchors: ['left', 'top'],
                    x: c * TILE_SIZE,
                    y: r * TILE_SIZE,
                    width: `${ TILE_SIZE } px`,
                    height: `${ TILE_SIZE } px`,
                    fallbackColor: '#444444',
                    onClick: () => this.handleTileClick(c, r),
                }));
            }
        }

        // Draw Blocks
        this.state.grid.forEach(key => {
            const [c, r] = key.split(',').map(Number);
            const block = gameContainer.children[0].addChild(new Drawable({
                anchors: ['left', 'top'],
                x: c * TILE_SIZE,
                y: r * TILE_SIZE,
                width: `${ TILE_SIZE } px`,
                height: `${ TILE_SIZE } px`,
                fallbackColor: '#aaaaaa', // Concrete color
                image: new TextureInfo(64, 64, "minigame/concrete"),
            }));
        });

        // Draw Water
        // Clip height based on waterLevel
        const waterHeight = this.state.waterLevel * totalHeight;
        
        const water = gameContainer.children[0].addChild(new Drawable({
            anchors: ['left', 'top'],
            x: 0,
            y: totalHeight - waterHeight,
            width: "100%",
            height: `${ waterHeight } px`,
            fallbackColor: '#00aaff', // Blue water
            image: new TextureInfo(64, 64, "minigame/water"),
            reddize: this.state.waterLevel > 0.8 // Warning color near top
        }));

        // Draw City Center (The Goal)
        const cityCenter = gameContainer.children[0].addChild(new Drawable({
            anchors: ['top', 'centerX'],
            centerY: 0,
            y: 10,
            width: `${ totalWidth } px`,
            height: "40px",
            fallbackColor: '#ffdd00', // Gold/Yellow
            image: new TextureInfo(64, 64, "minigame/citycenter")
        }));

        // Water warning text if close
        if (this.state.waterLevel > 0.8) {
            gameContainer.children[0].addChild(new Drawable({
                anchors: ['centerX'],
                centerOnOwnX: true,
                y: totalHeight / 2,
                width: "100%",
                height: "32px",
                text: "CRITICAL!",
                textAnchor: "center",
                fallbackColor: '#ff0000',
                grayscale: false
            }));
        }

        // UI Overlay (Score, Timer, Money)
        const uiBar = parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: 10 - this.scroller.getScroll(),
            width: "min(100%, 600px)",
            height: "40px",
            fallbackColor: '#222222',
            children: [
                new Drawable({
                    anchors: ['left', 'top'],
                    x: 10,
                    y: 10,
                    width: "100px",
                    height: "20px",
                    text: `Time: ${ this.state.timer } s`,
                    textAnchor: "left",
                    fallbackColor: '#ffffff'
                }),
                new Drawable({
                    anchors: ['centerX'],
                    centerOnOwnX: true,
                    y: 10,
                    width: "100px",
                    height: "20px",
                    text: `Score: ${ this.state.score } `,
                    textAnchor: "center",
                    fallbackColor: '#00ff00'
                }),
                new Drawable({
                    anchors: ['right', 'top'],
                    rightAlign: true,
                    x: 10,
                    y: 10,
                    width: "100px",
                    height: "20px",
                    text: `Money: ${ this.state.concrete } `,
                    textAnchor: "right",
                    fallbackColor: '#ffffff'
                })
            ]
        }));
    }

    private drawEndOverlay(parent: Drawable): void {
        const overlay = parent.addChild(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            width: "min(100%, 400px)",
            height: "100%",
            fallbackColor: 'rgba(0,0,0,0.8)',
            children: [
                new Drawable({
                    anchors: ['centerX'],
                    centerOnOwnX: true,
                    y: 100,
                    width: "300px",
                    height: "32px",
                    text: "Game Over",
                    textAnchor: "center",
                    fallbackColor: '#ffffff'
                }),
                new Drawable({
                    anchors: ['centerX'],
                    centerOnOwnX: true,
                    y: 150,
                    width: "300px",
                    height: "24px",
                    text: this.state.endReason,
                    textAnchor: "center",
                    fallbackColor: '#aaaaaa'
                }),
                new Drawable({
                    anchors: ['centerX'],
                    centerOnOwnX: true,
                    y: 200,
                    width: "300px",
                    height: "24px",
                    text: `Final Score: ${ this.state.score } `,
                    textAnchor: "center",
                    fallbackColor: '#ffffff'
                })
            ]
        }));

        overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: 260,
            width: "120px",
            height: "40px",
            fallbackColor: '#444444',
            onClick: () => { this.state.gameStarted = false; }, // Go back to menu
            children: [
                new Drawable({
                    anchors: ["centerX"],
                    y: 10,
                    width: "calc(100% - 10px)",
                    height: "100%",
                    text: "Return to Menu",
                    centerOnOwnX: true
                })
            ]
        }));
    }

    private drawRulesOverlay(root: Drawable, parent: Drawable): void {
        const overlay = root.addChild(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            width: "min(100%, 500px)",
            height: "100%",
            fallbackColor: '#333333',
            children: []
        }));

        overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: 10,
            width: "100%",
            height: "48px",
            text: "How to Play",
        }));

        // Rules Content
        let currentY = 80;
        const lines = [
            "1. The water is rising from the bottom.",
            "2. Click tiles to place Concrete Dykes.",
            "3. Each Dyke costs 15 Concrete.",
            "4. Water rises over time and resets the timer.",
            "5. If water touches the top City Center, you lose.",
            "6. Survive as long as possible to maximize your score!",
            "7. Practice mode is free but rewards nothing."
        ];

        lines.forEach(line => {
            const text = overlay.addChild(new Drawable({
                anchors: ['centerX'],
                centerOnOwnX: true,
                y: currentY,
                width: "calc(100% - 40px)",
                height: "24px",
                text: line,
                textAnchor: "center"
            }));
            currentY += 28;
        });

        // Close Button
        overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: currentY + 20,
            width: "120px",
            height: "40px",
            fallbackColor: '#666666',
            onClick: () => { this.toggleRules(); },
            children: [
                new Drawable({
                    anchors: ["centerX"],
                    y: 10,
                    width: "calc(100% - 10px)",
                    height: "100%",
                    text: "Close",
                    centerOnOwnX: true
                })
            ]
        }));

        // Scroll handling for rules
        root.onClick = () => { this.toggleRules(); };
    }
}
