﻿import { City } from "../game/City.js";
import { Flunds } from "../game/ResourceTypes.js";
import { TextureInfo } from "../ui/TextureInfo.js";
import { Drawable } from "../ui/Drawable.js";
import { IHasDrawable } from "../ui/IHasDrawable.js";
import { IOnResizeEvent } from "../ui/IOnResizeEvent.js";
import { StandardScroller } from "../ui/StandardScroller.js";
import { rangeMapExp } from "./MinigameUtil.js";

// --- Configuration & Constants ---

const CONFIG = {
    GAME_DURATION: 60, // seconds
    PLAYER_WIDTH: 80,
    PLAYER_HEIGHT: 15,
    PLAYER_SPEED: 1500, // pixels per second
    SPAWN_RATE_BASE: 1200, // ms
    SPAWN_RATE_MIN: 400,
    SPAWN_RATE_VARIANCE: 800,
    COST: { type: "wood", amount: 100 }
};

// --- Game Elements ---

// Types of falling items
const ITEM_TYPES = [
    { type: "good", name: "Lumber", color: "#D2691E", score: 10, weight: 40 }, // Good: Wood
    { type: "good", name: "Stone", color: "#808080", score: 15, weight: 30 }, // Good: Stone
    { type: "bad", name: "Bomb", color: "#000000", score: -50, weight: 10 },  // Bad: Bomb
    { type: "bonus", name: "Gold", color: "#FFD700", score: 50, weight: 5 },   // Bonus: Gold
    { type: "bonus", name: "Mystery", color: "#800080", score: -20, weight: 5 } // Bonus: Mystery (Bad)
];

// --- Helper Functions ---

/**
 * Randomly selects an item type based on weights.
 */
function getRandomItemType(): { type: string; name: string; color: string; score: number; weight: number } {
    const totalWeight = ITEM_TYPES.reduce((sum, item) => sum + item.weight, 0);
    let random = Math.random() * totalWeight;
    for (const item of ITEM_TYPES) {
        if (random < item.weight) return item;
        random -= item.weight;
    }
    return ITEM_TYPES[0];
}

// --- Minigame Class ---

export class WarehouseMinigame implements IHasDrawable, IOnResizeEvent {
    // --- Fields ---

    // Core Interfaces
    uiManager: any; // UIManager
    city: City;

    // State
    private shown: boolean;
    private gameStarted: boolean;
    private lastDrawable: Drawable | null = null;
    private uiResizeTimeout: number | NodeJS.Timeout | null = null;

    // Game Loop
    private lastTime: number = 0;
    private accumulator: number = 0;
    private lastSpawnTime: number = 0;
    private currentSpawnRate: number = CONFIG.SPAWN_RATE_BASE;

    // State Object
    private state = {
        gameState: 'IDLE' as 'IDLE' | 'PLAYING' | 'GAME_OVER', // IDLE, PLAYING, GAMEOVER
        playerX: 0, // Center X
        playerWidth: CONFIG.PLAYER_WIDTH,
        playerHeight: CONFIG.PLAYER_HEIGHT,

        score: 0,
        lives: 3,
        timeLeft: CONFIG.GAME_DURATION,

        items: Array<{ id: number; x: number; y: number; type: string; name: string; color: string; speed: number; score: number }>()
    };

    // Inputs
    private isDragging: boolean = false;
    private inputTargetX: number | null = null;

    // Assets
    private preloaded: boolean = false;
    private textures: { [key: string]: TextureInfo } = {};

    // Scroller
    private scroller: StandardScroller = new StandardScroller(false, true);

    constructor(city: City, uiManager: any) {
        this.city = city;
        this.uiManager = uiManager;
        this.shown = false;
    }

    // --- IHasDrawable Implementation ---

    asDrawable(): Drawable {
        if (!this.shown) {
            this.lastDrawable = new Drawable({ width: "0px" });
            return this.lastDrawable;
        }

        // Main container
        const root = new Drawable({
            width: "100%",
            height: "100%",
            fallbackColor: '#2c3e50' // Dark blue-grey background
        });

        if (this.state.gameState === 'IDLE') {
            this.drawStartScreen(root);
        } else if (this.state.gameStarted) {
            this.drawGameArea(root);
        } else {
            this.drawGameOver(root);
        }

        this.lastDrawable = root;
        return root;
    }

    getLastDrawable(): Drawable | null {
        return this.lastDrawable;
    }

    // --- IOnResizeEvent Implementation ---

    onResize(): void {
        if (this.uiResizeTimeout) {
            clearTimeout(this.uiResizeTimeout);
        }
        // Debounce resize handling
        this.uiResizeTimeout = setTimeout(() => {
            this.scroller.onResize();
            this.uiResizeTimeout = null;
        }, 100);
    }

    // --- Game Flow ---

    preloadImages(): void {
        if (this.preloaded) return;
        // In a real scenario, these would be real assets.
        // Here we just assume they are loaded or generate programmatic ones.
        this.preloaded = true;
    }

    show(): void {
        this.shown = true;
        this.uiManager.frameRequested = true;
    }

    hide(): void {
        this.shown = false;
        this.uiManager.frameRequested = true;
    }

    private startGame(): void {
        // Check costs
        if (!this.city.hasResources([CONFIG.COST], false)) {
            return; // Alert/Notify is handled by main game loop or specific UI feedback
        }

        // Deduct costs
        this.city.transferResourcesFrom([{ type: CONFIG.COST.type, amount: CONFIG.COST.amount }], "earn");

        // Initialize State
        this.state.gameStarted = true;
        this.state.score = 0;
        this.state.lives = 3;
        this.state.timeLeft = CONFIG.GAME_DURATION;
        this.state.items = [];
        this.state.gameState = 'PLAYING';

        // Center player
        this.state.playerX = 0; // Will be set in resize

        // UI Updates
        this.uiManager.frameRequested = true;
    }

    private endGame(): void {
        this.state.gameState = 'GAME_OVER';

        // Calculate Winnings
        // Basic logic: Score determines resources.
        // Let's say Score >= 500 gets a bonus.
        const winnings: { type: string; amount: number }[] = [
            { type: "wood", amount: Math.max(0, Math.floor(this.state.score / 10)) }
        ];

        if (this.state.score > 1000) {
            winnings.push({ type: "stone", amount: Math.floor((this.state.score - 1000) / 20) });
        }

        if (winnings[0].amount === 0 && this.state.score < 50) {
            // No point playing if you get nothing and score is low
            // But we still give the wood
        }

        this.city.transferResourcesFrom(winnings, "earn");

        this.uiManager.frameRequested = true;
    }

    // --- Update Logic ---

    update(dt: number): void {
        if (!this.shown) return;

        // Game Loop if playing
        if (this.state.gameState === 'PLAYING') {
            // Timer
            this.state.timeLeft -= dt;
            if (this.state.timeLeft <= 0) {
                this.state.timeLeft = 0;
                this.endGame();
                return;
            }

            // Spawning
            this.lastSpawnTime += dt * 1000;
            if (this.lastSpawnTime > this.currentSpawnRate) {
                this.spawnItem();
                this.lastSpawnTime = 0;

                // Increase difficulty
                if (this.currentSpawnRate > CONFIG.SPAWN_RATE_MIN) {
                    this.currentSpawnRate -= 5; // Make it faster
                }
            }

            // Player Movement (Lerp towards target)
            if (this.inputTargetX !== null) {
                // Smooth movement
                const diff = this.inputTargetX - this.state.playerX;
                if (Math.abs(diff) > 5) {
                    this.state.playerX += diff * dt * 10;
                } else {
                    this.state.playerX = this.inputTargetX;
                }
            }

            // Move items
            const moveSpeed = 100 + (this.state.score * 0.5); // Get faster as score goes up
            for (let i = this.state.items.length - 1; i >= 0; i--) {
                const item = this.state.items[i];
                item.y += item.speed * dt;

                // Collision Detection
                // AABB Collision
                if (
                    item.y + 20 > this.state.playerX - this.state.playerWidth / 2 &&
                    item.y < this.state.playerX + this.state.playerWidth / 2
                ) {
                    // Check Y overlap (approximate for falling down)
                    if (Math.abs(item.y - (this.state.playerX - this.state.playerWidth / 2)) < 10) {
                        this.handleCollision(item);
                        this.state.items.splice(i, 1);
                        continue;
                    }
                }

                // Remove if off screen
                if (item.y > 800) { // 800 is approx screen height
                    this.state.items.splice(i, 1);
                }
            }
        }
    }

    private handleCollision(item: any) {
        this.state.score += item.score;
        if (this.state.score < 0) this.state.score = 0;
    }

    private spawnItem(): void {
        if (this.state.items.length > 5) return; // Limit count

        const itemType = getRandomItemType();
        // Random X position
        const x = Math.random() * 100; // 0-100% relative
        const y = -50;

        this.state.items.push({
            id: Date.now(),
            x: x,
            y: y,
            type: itemType.type,
            name: itemType.name,
            color: itemType.color,
            speed: 100 + Math.random() * 50,
            score: itemType.score
        });
    }

    // --- Drawing ---

    private drawStartScreen(parent: Drawable): void {
        const overlay = parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            width: "100%",
            height: "100%",
            fallbackColor: '#2c3e50',
            onDrag: (x: number, y: number) => {
                this.scroller.handleDrag(y, { height: 0, width: 0, x: 0, y: 0 }); // Hack for scrolling
            },
            onDragEnd: () => {
                this.scroller.resetDrag();
            }
        }));

        const content = overlay.addChild(new Drawable({
            x: 0,
            y: 100 - this.scroller.getScroll(),
            width: "calc(100% - 40px)",
            height: "100%",
            fallbackColor: 'transparent'
        }));

        // Header
        content.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: 50,
            width: "calc(100% - 60px)",
            height: "80px",
            text: "Logistics Rush",
            textAnchor: 'middle',
            fontSize: '32px',
            color: '#ecf0f1',
            fallbackColor: 'transparent'
        }));

        // Rules
        let nextY = 150;
        const addRule = (text: string) => {
            content.addChild(new Drawable({
                anchors: ['centerX'],
                centerOnOwnX: true,
                y: nextY,
                width: "calc(100% - 60px)",
                height: "auto",
                text: text,
                color: '#bdc3c7',
                wordWrap: true,
                keepParentWidth: true,
                fallbackColor: 'transparent'
            }));
            nextY += 40;
        };

        addRule("Catch the crates to get Wood and Stone.");
        addRule("Avoid the Bombs!");
        addRule("Collect Gold for bonus points.");

        nextY += 50;

        // Start Button
        const canAfford = this.city.hasResources([CONFIG.COST], false);
        const startButton = content.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "200px",
            height: "60px",
            text: "Start Shift",
            fontSize: '24px',
            color: '#ffffff',
            fallbackColor: canAfford ? '#27ae60' : '#7f8c8d',
            onClick: () => {
                if (canAfford) {
                    this.startGame();
                }
            }
        }));

        // Cost
        const costs = [CONFIG.COST];
        content.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY + 80,
            width: "calc(100% - 60px)",
            text: `Cost: ${CONFIG.COST.amount} ${CONFIG.COST.type.toUpperCase()}`,
            color: canAfford ? '#f1c40f' : '#e74c3c',
            textAnchor: 'center',
            fallbackColor: 'transparent'
        }));

        // Scroll size
        this.scroller.setChildrenSize(nextY + 200);
    }

    private drawGameArea(parent: Drawable): void {
        // Background
        parent.fallbackColor = '#2c3e50';

        const gameArea = parent.addChild(new Drawable({
            x: 0,
            y: 0,
            width: "100%",
            height: "100%"
        }));

        // HUD
        // 1. Score
        gameArea.addChild(new Drawable({
            anchors: ['left'],
            y: 10,
            x: 20,
            text: `Score: ${Math.floor(this.state.score)}`,
            color: '#ecf0f1',
            fontSize: '20px',
            fallbackColor: 'transparent'
        }));

        // 2. Lives (Hearts)
        let heartsText = "";
        for (let i = 0; i < this.state.lives; i++) heartsText += "❤️";
        gameArea.addChild(new Drawable({
            anchors: ['right'],
            y: 10,
            x: -20,
            rightAlign: true,
            text: heartsText,
            fontSize: '20px',
            fallbackColor: 'transparent'
        }));

        // 3. Timer
        gameArea.addChild(new Drawable({
            anchors: ['top'],
            y: 10,
            x: '50%',
            centerOnOwnX: true,
            text: `${Math.ceil(this.state.timeLeft)}s`,
            color: '#ecf0f1',
            fontSize: '20px',
            fallbackColor: 'transparent'
        }));

        // 4. Player (Platform)
        // We use a simple rect for now, but in a real game we would use the texture
        const player = gameArea.addChild(new Drawable({
            anchors: ['centerX'],
            y: 0, // Will be set in update logic
            x: this.state.playerX,
            width: `${CONFIG.PLAYER_WIDTH}px`,
            height: `${CONFIG.PLAYER_HEIGHT}px`,
            fallbackColor: '#3498db',
            text: '🚜', // Tractor emoji as placeholder texture
            textAlign: 'center',
            lineHeight: CONFIG.PLAYER_HEIGHT
        }));

        // Save reference for drawing updates
        // (In a real engine, we wouldn't modify the drawable tree every frame, 
        // but for this spec's simplicity, we do it this way for the "asDrawable" pattern)
        const playerRef = player;

        // 5. Items
        this.state.items.forEach(item => {
            const itemDrawable = gameArea.addChild(new Drawable({
                anchors: ['centerX'],
                y: item.y,
                x: item.x,
                width: "40px",
                height: "40px",
                fallbackColor: item.color,
                text: item.type === 'good' ? '📦' : (item.type === 'bad' ? '💣' : '💎'),
                textAlign: 'center',
                lineHeight: '40px',
                fontSize: '20px'
            }));
        });

        // 6. Inputs (Gameplay)
        // We attach the handler to the game area
        gameArea.onDrag = (x: number, y: number) => {
            // x is percentage (0-1) or normalized? Usually normalized for UI.
            // Assuming x is normalized 0-1 based on screen width
            const width = 100; // %
            this.state.playerX = x;
            this.state.playerX = Math.max(0, Math.min(100, this.state.playerX)); // Clamp 0-100
        };

        // We need a way to update the player position in the tree if the tree existed as an object.
        // Since asDrawable() rebuilds the tree, we can't easily update children properties without rebuilding.
        // However, the spec says "asDrawable returns the drawable".
        // Wait, if we rebuild the tree, we lose the 'onDrag' handler we just set.
        // The pattern implies we rebuild the tree. 
        // This is a common pattern in this specific engine (rebuild on every frame or state change).
        // But wait, if we rebuild, how does the engine keep the 'onDrag' attached?
        // Ah, the engine likely calls asDrawable, draws it, and if the pointer is down, it calls the onDrag.
        // If we rebuild the tree, the onDrag is on the new item.
        // So we just need to make sure we don't lose the reference to the active game area drawable.

        // Actually, looking at the spec: "onDrag: (x: number, y: number) => void | null".
        // This suggests the engine iterates the tree or we set the last one.
        // Given the constraints, I will just rebuild. The engine usually handles the event wiring.
        // I will assume the engine handles the event wiring or I have to do it.
        // Let's assume the engine handles the "current" drawable's events.
    }

    private drawGameOver(parent: Drawable): void {
        const overlay = parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            width: "100%",
            height: "100%",
            fallbackColor: 'rgba(0,0,0,0.8)',
            onDrag: () => { } // Block interaction
        }));

        overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: 200,
            text: "Shift Ended",
            color: '#ecf0f1',
            fontSize: '40px',
            fallbackColor: 'transparent'
        }));

        const scoreText = `Final Score: ${Math.floor(this.state.score)}`;
        overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: 260,
            text: scoreText,
            color: '#f1c40f',
            fontSize: '24px',
            fallbackColor: 'transparent'
        }));

        overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: 320,
            text: "Click to return",
            color: '#95a5a6',
            fontSize: '18px',
            fallbackColor: 'transparent',
            onClick: () => {
                // Return to idle
                this.shown = false;
                this.state.gameState = 'IDLE';
                // Trigger a redraw
                this.uiManager.frameRequested = true;
            }
        }));
    }

    // Get cost for the main menu
    getCosts(): { type: string, amount: number }[] {
        return [CONFIG.COST];
    }
}
