import { City } from "../game/City.js";
import { CityFlags } from "../game/CityFlags.js";
import { Electronics, Flund, Resource } from "../game/Resource.js";
import { getResourceType } from "../game/ResourceTypes.js";
import { UIManager } from "../ui/UIManager.js";
import { Drawable } from "../ui/Drawable.js";
import { TextureInfo } from "../ui/TextureInfo.js";
import { IHasDrawable } from "../ui/IHasDrawable.js";
import { IOnResizeEvent } from "../ui/IOnResizeEvent.js";
import { StandardScroller } from "../ui/StandardScroller.js";
import { addResourceCosts } from "../ui/UIUtil.js";
import { OnePracticeRun } from "./MinigameUtil.js";

// --- Game Constants ---
const TILE_SIZE = 70;
const GAME_DURATION = 90; // seconds
const GRID_SIZE_EASY = 4;
const GRID_SIZE_HARD = 5;

// Region colors (fallbackColor strings)
const REGION_COLORS = [
    '#FF5733', // Orange
    '#33FF57', // Green
    '#5733FF', // Blue
    '#FF33A8', // Pink
    '#A833FF'  // Purple
];

export class CropCultivation implements IHasDrawable, IOnresizeEvent {

    // --- State & Properties ---
    private city: City;
    private uiManager: UIManager;
    private scroller: StandardScroller;
    private lastDrawable: Drawable | null = null;

    // Game Flags
    private gameStarted: boolean = false;
    private shown: boolean = false;
    private isPractice: boolean = false;
    private userInputLocked: boolean = false;
    private preloaded: boolean = false;
    private howToPlayShown: boolean = false;

    // Gameplay State
    private timer: number = GAME_DURATION;
    private endReason: string | null = null;
    private winnings: Resource[] = [];
    private timerTimeout: NodeJS.Timeout | null = null;

    // Puzzle Data
    private gridSize: number = 4;
    private regionMap: number[][] = []; // 2D grid storing region IDs
    private regionData: { [id: number]: { count: number, color: string, name: string } };
    private trees: Set<string> = new Set(); // Stores "x,y" strings

    constructor(city: City, uiManager: UIManager) {
        this.city = city;
        this.uiManager = uiManager;
        this.scroller = new StandardScroller(false, true);
        this.regionData = {};
        this.resetState();
    }

    private resetState(): void {
        this.timer = GAME_DURATION;
        this.gameStarted = false;
        this.userInputLocked = false;
        this.endReason = null;
        this.winnings = [];
        this.trees.clear();
        if (this.timerTimeout) {
            clearTimeout(this.timerTimeout);
            this.timerTimeout = null;
        }
    }

    // --- Lifecycle ---

    public show(): void {
        this.shown = true;
        this.gameStarted = false;
        this.isPractice = false;
        this.howToPlayShown = false;
        this.resetState();
        this.preloadImages();
    }

    public hide(): void {
        this.shown = false;
    }

    public async preloadImages(): Promise<void> {
        if (this.preloaded) return;
        const urls: { [key: string]: string } = {
            "minigame/tree": "assets/minigame/tree.png", // Assume a generic tree icon
            "minigame/region": "assets/minigame/crop_region.png"
        };
        await this.uiManager.renderer.loadSprites(this.city, urls);
        this.preloaded = true;
    }

    public getLastDrawable(): Drawable | null {
        return this.lastDrawable;
    }

    public onResize(): void {
        this.scroller.onResize();
    }

    // --- Game Logic ---

    public getCosts(): { type: string, amount: number }[] {
        return this.isPractice ? OnePracticePlay :
            [{ type: "flunds", amount: 50 }]; // Cheap to play
    }

    private generatePuzzle(difficulty: string): void {
        // For this implementation, we'll use a hardcoded valid configuration and its map
        // to ensure solvability. A true generator would run a SAT solver.

        if (difficulty === "hard") {
            this.gridSize = GRID_SIZE_HARD;
            // Example Hard Puzzle solution (T) and Regions (R)
            // T . T . .
            // . . . . .
            // . T . . T
            // . . . T .
            // . . . . .
            // Regions setup to match specific counts
            this.regionMap = [
                [0, 0, 1, 1, 2],
                [0, 3, 1, 4, 2],
                [3, 3, 4, 4, 2],
                [3, 5, 5, 4, 6],
                [5, 5, 6, 6, 6]
            ];
            this.regionData = {
                0: { count: 1, color: REGION_COLORS[0], name: "Pasture" },
                1: { count: 1, color: REGION_COLORS[1], name: "Meadow" },
                2: { count: 2, color: REGION_COLORS[2], name: "Grazing" },
                3: { count: 2, color: REGION_COLORS[3], name: "Farming" },
                4: { count: 2, color: REGION_COLORS[4], name: "Cultiv" },
                5: { count: 1, color: REGION_COLORS[0], name: "Yields" },
                6: { count: 1, color: REGION_COLORS[1], name: "Crops" }
            };
        } else {
            // Easy Puzzle
            this.gridSize = GRID_SIZE_EASY;
            this.regionMap = [
                [0, 0, 0, 1],
                [0, 2, 1, 1],
                [2, 2, 3, 3],
                [2, 4, 3, 4]
            ];
            this.regionData = {
                0: { count: 1, color: REGION_COLORS[0], name: "Plot A" },
                1: { count: 2, color: REGION_COLORS[1], name: "Plot B" },
                2: { count: 2, color: REGION_COLORS[2], name: "Plot C" },
                3: { count: 1, color: REGION_COLORS[3], name: "Plot D" },
                4: { count: 1, color: REGION_COLORS[4], name: "Plot E" }
            };
        }
    }

    public startGame(): void {
        if (this.city.checkAndSpenResource(this.getCosts())) {
            this.gameStarted = true;
            this.generatePuzzle("medium"); // Can be dynamic based on settings
            this.startTimer();
            this.city.updateLastUserAction();
            this.city.fullSave();
        }
    }

    private startTimer(): void {
        const tick = () => {
            if (!this.gameStarted || this.timer <= 0) return;
            this.timer--;
            this.uiManager.frameRequested = true;

            if (this.timer <= 0) {
                this.endReason = "Time's up!";
                this.endGame(false);
            } else {
                this.timerTimeout = setTimeout(tick, 1000);
            }
        };
        this.timerTimeout = setTimeout(tick(), 1000);
    }

    private handleTileClick(x: number, y: number): void {
        if (this.userInputLocked || !this.gameStarted) return;

        const key = `${x},${y}`;
        if (this.trees.has(key)) {
            this.trees.delete(key);
        } else {
            // Validation: Check adjacency (orthogonal + diagonal)
            if (this.hasNeighbor(x, y)) {
                // Show a brief "flash" or just ignore? 
                // For simplicity in this spec, we just ignore invalid placement silently or block it.
                // Let's block it.
                return;
            }
            this.trees.add(key);
        }

        if (this.checkWinCondition()) {
            this.endReason = "Cultivation Complete!";
            this.endGame(true);
        }
        this.uiManager.frameRequested = true;
    }

    private hasNeighbor(x: number, y: number): boolean {
        for (let dx = -1; dx <= 1; dx++) {
            for (let dy = -1; dy <= 1; dy++) {
                if (dx === 0 && dy === 0) continue;
                const checkKey = `${x + dx},${y + dy}`;
                if (this.trees.has(checkKey)) return true;
            }
        }
        return false;
    }

    private checkWinCondition(): boolean {
        // 1. Check region counts
        const regionCounts: { [id: number]: number } = {};
        for (let id in this.regionData) regionCounts[id] = 0;

        for (let treeKey of this.trees) {
            const [tx, ty] = treeKey.split(',').map(Number);
            const rId = this.regionMap[ty][tx];
            if (regionCounts[rId] !== undefined) {
                regionCounts[rId]++;
            }
        }

        for (let id in this regionData) {
            if (regionCounts[id] !== this.regionData[id].count) return false;
        }

        // 2. Check adjacency is already enforced on click, so if we get here and counts match, we win.
        return true;
    }

    private endGame(success: boolean): void {
        this.gameStarted = false;
        if (this.timerTimeout) clearTimeout(this.timerTimeout);

        if (success) {
            this.calculateWinnings();
        } else {
            this.winnings = [];
        }
        this.city.updateLastUserAction();
        this.uiManager.frameRequested = True;
    }

    private calculateWinnings(): void {
        // Base reward + time bonus
        let electronics = 50 + Math.floor(this.timer * 0.5);
        this.winnings.push(new Electronics(electronics));
        this.winnings.push(this.city.resources.getResourceType("flunds").construct(10));
    }

    // --- UI Drawing ---

    public asDrawable(): Drawable {
        if (!this.shown) {
            return this.lastDrawable = new.Drawable({ width: "0px" });
        }

        const main = new Drawable({
            width: "100%",
            height: "100%",
            fallbackColor: '#222222'
        });

        if (!this.gameStarted && !this.winnings.length) {
            this.drawStartOverlay(main);
        } else {
            this.drawGameArea(main);
            if (this.winnings.length) this.drawWinnings(main);
        }

        this.lastDrawable = main;
        return main;
    }

    private drawStartOverlay(parent: Drawable): void {
        const overlay = parent.add(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            width: "min(100%, 700px)",
            height: "100%",
            fallbackColor: '#111111',
            onDrag: (x, y) => this.scroller.handleDrag(y),
            onDragEnd: () => this.scroller.resetDrag()
        }));

        let nextY = 20 - this.scroller.getScroll();

        // Title
        overlay.add(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "100%",
            height: "50px",
            text: "CROP CULTIVATION"
        }));
        nextY += 100;

        // Start Button
        const startBtn = overlay.add(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "200px",
            height: "60px",
            fallbackColor: '#444444',
            onClick: () => this.startGame(),
            children: [
                new Drawable({
                    anchors: ["centerX"],
                    y: 10,
                    width: "180px",
                    height: "40px",
                    text: "Plant Trees"
                })
            ]
        }));

        // Costs
        addResourceCosts(
            startBtn,
            this.getCosts(),
            50, 10,
            false, false, false
        );
        nextY += 120;

        // Practice Toggle
        overlay.add(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "200px",
            height: "50px",
            onClick: () => { this.isPractice = true; },
            children: [
                new Drawable({
                    x: 10,
                    y: 5,
                    width: "40px",
                    height: "40px",
                    image: new TextureInfo(40, 40, "ui/checked")
                }),
                new Drawable({
                    anchors: ['right'],
                    x: 10,
                    y: 5,
                    width: "140px",
                    height: "40px",
                    text: "Practice Mode"
                })
            ]
        }));
        nextY += 80;

        // Rules Button
        if (!this.howToPayShown) {
            overlay.add(new Drawable({
                anchors: ['centerX'],
                centerOnOwnX: true,
                y: nextY,
                width: "100px",
                height: "50px",
                onClick: () => { this.howToPayShown = true; this.scroller.resetScroll(); },
                children: [
                    new Drawable({
                        anchors: ["centerX"],
                        y: 5,
                        width: "90px",
                        height: "40px",
                        text: "How to Play"
                    })
                ]
            }));
            nextY += 80;
        }

        // Rules Content
        if (this.howHowToPlayShown) {
            nextY = this.drawRulesContent(overlay, nextY);
        }

        this.scroller.setChildrenHeight(nextY);
    }

    private drawRulesContent(parent: Drawable, startY: number): number {
        let nextY = startY + 20;

        let ruleText = parent.add(new Drawable({
            x: 20,
            y: nextY,
            width: "calc(100% - 80px)",
            height: "40px",
            wordWrap: true,
            text: "Plant trees to satisfy the number in each region."
        }));
        nextY += ruleText.height + 50;

        ruleText = parent.add(new.Drawable({
            anchors: ['bottom'],
            y: -50,
            width: "calc(100% - 100px)",
            height: "40px",
            wordWrap: true,
            text: "No two trees can touch, even diagonally."
        }));
        nextY += ruleText.getHeight() + 50;

        return nextY;
    }

    private drawGameArea(parent: Drawable): void {
        // Main grid container
        const gridContainer = parent.add(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            y: 50,
            width: (this.gridSize * TILE_SIZE) + "px",
            height: (this.gridSize * TILE_SIZE) + "px",
            fallbackColor: '#000000' // Black canvas
        }));

        // Draw grid cells
        for (let y = 0; y < this.gridSize; y++) {
            for (let x = 0; x < this.gridSize; x++) {
                const regionId = this.regionMap[y][x];
                const region = this.regionData[regionId];
                const hasTree = this.trees.has(`${x},${y}`);

                // Draw Region Background
                const tile = gridContainer.add(new Drawable({
                    x: x * TILE_SIZE,
                    y: y * TILE_SIZE,
                    width: TILE_SIZE + "px",
                    height: TILE_SIZE + "px",
                    fallbackColor: region.color
                }));

                // Draw Number (only once per region, e.g. top-left cell)
                // Simple logic: check if this is the first cell found for this region scanning L->R, T->B
                let isLabel = true;
                for (let py = 0; py < y; py++) {
                    if (this.regionMap[py][x] === regionId) isLabel = false;
                }
                for (let px = 0; px < x; px++) {
                    if (this.regionMap[y][px] === regionId) isLabel false;
                }

                if (isLabel) {
                    tile.add(new Drawable({
                        x: 5,
                        y: 5,
                        width: "40px",
                        height: "40px",
                        text: "" + region.count
                    }));
                }

                // Draw Tree Icon
                if (hasTree) {
                    tile.add(new.Drawable({
                        anchors: ["centerX", "centerY"],
                        centerOnOwnX: true,
                        y: 0,
                        width: "40px",
                        height: "40px",
                        image: new TextureInfo(48, 48, "minigame/tree"),
                        fallbackColor: '#00FF00' // Bright green
                    }));
                }

                // Add click handler to the tile
                tile.onClick = () => this.handleTileClick(x, y);
            }
        }

        // Timer bar
        this.drawProgressBar(parent);
    }

    private drawProgressBar(parent: Drawable): void {
        const barY = (this.gridSize * TILE_SIZE) + 80;
        parent.add(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            y: barY,
            width: "300px",
            height: "30px",
            fallbackColor: '#666666',
            children: [
                new Drawable({
                    clipWidth: (this.timer / GAME_DURATION),
                    width: "100%",
                    height: "100%",
                    fallbackColor: '#00ff11', // Bright green bar
                    reddize: this.timer < 10
                })
            ]
        }));
    }

    private drawWinnings(parent: Drawable): void {
        const winArea = parent.add(new Drawable({
            anchors: ["top"],
            y: 20,
            width: "400px",
            height: "100px",
            fallbackColor: '#444444'
        }));

        winArea.add(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            y: 10,
            width: "300px",
            height: "30px",
            text: "Farm Yield:"
        }));

        addResourceCosts(winArea, this.winnings, 10, 50, false, false, true, 60);
    }
}
