// ResourceSorter.ts
import { City } from "../game/City.js";
import { CityFlags } from "../game/CityFlags.js";
import { Resource } from "../game/Resource.js";
import { Flunds, getResourceType } from "../game/ResourceTypes.js";
import { TextureInfo } from "../ui/TextureInfo.js";
import { Drawable } from "../ui/Drawable.js";
import { UIManager } from "../ui/UIManager.js";
import { IHasDrawable } from "../ui/IHasDrawable.js";
import { IOnResizeEvent } from "../ui/IOnResizeEvent.js";
import { StandardScroller } from "../ui/StandardScroller.js";
import { addResourceCosts, humanizeFloor } from "../ui/UIUtil.js";
import { OnePracticeRun, progressMinigameOptionResearch } from "./MinigameUtil.js";
import { EffectType } from "../game/GridType.js";

// Game constants
const GAME_DURATION = 60; // seconds
const TILE_SIZE = 64;
const CONTAINER_SIZE = 80;
const RESOURCE_DROP_INTERVAL = 2000; // milliseconds
const MAX_RESOURCES_PER_CONTAINER = 10;

// Resource types for this minigame
const RESOURCE_TYPES = [
    "Food", "BuildingMaterials", "Fuel", "ManufacturedGoods"
] as const;
type ResourceType = typeof RESOURCE_TYPES[number];

interface ResourceItem {
    type: ResourceType;
    x: number;
    y: number;
    falling: boolean;
    containerIndex: number | null;
}

interface Container {
    type: ResourceType;
    x: number;
    resources: number;
    maxResources: number;
}

interface ResourceSorterState {
    score: number;
    timer: number;
    resources: ResourceItem[];
    containers: Container[];
    currentResource: ResourceType | null;
    gameStarted: boolean;
    userInputLocked: boolean;
    endReason: string | null;
}

export class ResourceSorter implements IHasDrawable, IOnResizeEvent {
    private state: ResourceSorterState;
    private scroller: StandardScroller;
    private resourceTimeout: NodeJS.Timeout | null = null;
    private timerTimeout: NodeJS.Timeout | null = null;
    private preloaded: boolean = false;
    private endReason: string | null = null;
    private winnings: Flunds[] | null = null;
    private isPractice: boolean = false;
    private shown: boolean = false;
    private lastDrawable: Drawable | null = null;

    constructor(private city: City, private uiManager: UIManager) {
        this.state = {
            score: 0,
            timer: GAME_DURATION,
            resources: [],
            containers: this.initializeContainers(),
            currentResource: null,
            gameStarted: false,
            userInputLocked: false,
            endReason: null
        };
        this.scroller = new StandardScroller(false, true);
    }

    private initializeContainers(): Container[] {
        return RESOURCE_TYPES.map(type => ({
            type,
            x: 0,
            resources: 0,
            maxResources: MAX_RESOURCES_PER_CONTAINER
        }));
    }

    public async preloadImages(): Promise<void> {
        if (this.preloaded) return;

        const urls: { [key: string]: string } = {
            "minigame/sorter-container-food": "assets/minigame/sorter-container-food.png",
            "minigame/sorter-container-building": "assets/minigame/sorter-container-building.png",
            "minigame/sorter-container-fuel": "assets/minigame/sorter-container-fuel.png",
            "minigame/sorter-container-manufactured": "assets/minigame/sorter-container-manufactured.png",
            "minigame/sorter-resource-food": "assets/minigame/sorter-resource-food.png",
            "minigame/sorter-resource-building": "assets/minigame/sorter-resource-building.png",
            "minigame/sorter-resource-fuel": "assets/minigame/sorter-resource-fuel.png",
            "minigame/sorter-resource-manufactured": "assets/minigame/sorter-resource-manufactured.png"
        };

        await this.uiManager.renderer.loadMoreSprites(this.city, urls);
        this.preloaded = true;
    }

    public startGame(): void {
        if (!this.city.checkAndSpendResources(this.getCosts())) {
            return;
        }

        this.city.updateLastUserActionTime();
        this.city.game!.fullSave();

        this.state = {
            score: 0,
            timer: GAME_DURATION,
            resources: [],
            containers: this.initializeContainers(),
            currentResource: this.getRandomResourceType(),
            gameStarted: true,
            userInputLocked: false,
            endReason: null
        };

        this.startTimer();
        this.spawnResource();
    }

    private getRandomResourceType(): ResourceType {
        return RESOURCE_TYPES[Math.floor(Math.random() * RESOURCE_TYPES.length)];
    }

    private startTimer(): void {
        this.timerTimeout = setTimeout(() => {
            if (!this.state.gameStarted) return;

            this.state.timer--;
            this.uiManager.frameRequested = true;

            if (this.state.timer <= 0) {
                this.endReason = "Time's up!";
                this.endGame();
            } else {
                this.startTimer();
            }
        }, 1000);
    }

    private spawnResource(): void {
        if (!this.state.gameStarted || !this.state.currentResource) return;

        const newResource: ResourceItem = {
            type: this.state.currentResource,
            x: Math.random() * (this.uiManager.width - TILE_SIZE),
            y: -TILE_SIZE,
            falling: true,
            containerIndex: null
        };

        this.state.resources.push(newResource);
        this.state.currentResource = this.getRandomResourceType();
        this.uiManager.frameRequested = true;

        this.resourceTimeout = setTimeout(() => this.spawnResource(), RESOURCE_DROP_INTERVAL);
    }

    private handleResourceClick(resource: ResourceItem): void {
        if (this.state.userInputLocked || !this.state.gameStarted) return;

        // Find the matching container
        const containerIndex = this.state.containers.findIndex(c => c.type === resource.type);
        if (containerIndex === -1) return;

        const container = this.state.containers[containerIndex];
        if (container.resources >= container.maxResources) {
            this.state.endReason = "Container overflow!";
            this.endGame();
            return;
        }

        // Place resource in container
        resource.falling = false;
        resource.containerIndex = containerIndex;
        container.resources++;
        this.state.score += 10;

        // Remove resource from falling list after animation
        setTimeout(() => {
            this.state.resources = this.state.resources.filter(r => r !== resource);
            this.uiManager.frameRequested = true;
        }, 300);

        this.uiManager.frameRequested = true;
    }

    private endGame(): void {
        this.state.gameStarted = false;
        this.state.userInputLocked = true;

        if (this.resourceTimeout) {
            clearTimeout(this.resourceTimeout);
            this.resourceTimeout = null;
        }

        if (this.timerTimeout) {
            clearTimeout(this.timerTimeout);
            this.timerTimeout = null;
        }

        this.calculateWinnings();
    }

    private calculateWinnings(): void {
        const baseReward = Math.floor(this.state.score / 10);
        const flundsReward = new Flunds(baseReward);
        this.winnings = [flundsReward];

        // Add research progress as bonus
        progressMinigameOptionResearch(this.city, baseReward * 0.1);
    }

    public getCosts(): { type: string, amount: number }[] {
        return this.isPractice ? OnePracticeRun :
            [{ type: "flunds", amount: 100 }];
    }

    public asDrawable(): Drawable {
        if (!this.shown) return this.lastDrawable = new Drawable({ width: "0px" });

        const mainDrawable = new Drawable({
            width: "100%",
            height: "100%",
            fallbackColor: '#2c3e50'
        });

        if (!this.state.gameStarted) {
            this.drawStartOverlay(mainDrawable);
        } else {
            this.drawGameArea(mainDrawable);
        }

        this.lastDrawable = mainDrawable;
        return mainDrawable;
    }

    private drawStartOverlay(parent: Drawable): void {
        const overlay = parent.addChild(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            width: "min(100%, 600px)",
            height: "100%",
            fallbackColor: '#34495e',
            onDrag: (x: number, y: number) => { this.scroller.handleDrag(y, overlay.screenArea); },
            onDragEnd: () => { this.scroller.resetDrag(); },
        }));

        let nextY = 10 - this.scroller.getScroll();
        let baseY = nextY;

        // Title
        overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "100%",
            height: "48px",
            text: "Resource Sorter"
        }));
        nextY += 60;

        // Start button
        const startButton = overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "220px",
            height: "48px",
            fallbackColor: '#27ae60',
            onClick: () => this.startGame(),
            children: [
                new Drawable({
                    anchors: ["centerX"],
                    y: 5,
                    width: "calc(100% - 10px)",
                    height: "100%",
                    text: "Start Game",
                    centerOnOwnX: true
                })
            ]
        }));

        // Resource costs
        const unaffordable = !this.city.hasResources(this.getCosts(), false);
        addResourceCosts(startButton, this.getCosts(), 86, 58, false, false, false, 48, 10, 32, undefined, undefined, unaffordable, this.city);
        nextY += 60;

        // Practice mode toggle
        overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "220px",
            height: "48px",
            fallbackColor: '#3498db',
            onClick: () => { this.isPractice = !this.isPractice; },
            children: [
                new Drawable({
                    x: 5,
                    width: "48px",
                    height: "48px",
                    image: new TextureInfo(64, 64, this.isPractice ? "ui/checked" : "ui/unchecked"),
                }),
                new Drawable({
                    anchors: ["right"],
                    rightAlign: true,
                    x: 5,
                    y: 7,
                    width: "calc(100% - 60px)",
                    height: "100%",
                    text: "Practice Run (no rewards)",
                }),
            ]
        }));
        nextY += 60;

        // How to play button
        overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "220px",
            height: "48px",
            fallbackColor: '#e67e22',
            onClick: () => { this.toggleRules(); },
            children: [
                new Drawable({
                    anchors: ["centerX"],
                    y: 5,
                    width: "calc(100% - 10px)",
                    height: "100%",
                    text: "How to Play",
                    centerOnOwnX: true
                })
            ]
        }));
        nextY += 60;

        this.scroller.setChildrenSize(nextY - baseY);
    }

    private drawGameArea(parent: Drawable): void {
        let nextY = this.drawContainers(parent, 80);
        nextY = this.drawTimer(parent, nextY);
        nextY = this.drawScore(parent, nextY);
        nextY = this.drawNextResource(parent, nextY);

        // Draw falling resources
        this.state.resources.forEach(resource => {
            if (resource.falling) {
                const resourceDrawable = parent.addChild(new Drawable({
                    x: resource.x,
                    y: resource.y,
                    width: TILE_SIZE + "px",
                    height: TILE_SIZE + "px",
                    image: new TextureInfo(64, 64, `minigame/sorter-resource-${resource.type}`),
                    onClick: () => this.handleResourceClick(resource)
                }));

                // Animate falling
                if (resource.falling) {
                    resource.y += 2;
                    if (resource.y > this.uiManager.height - 100) {
                        this.state.endReason = "Resource hit the ground!";
                        this.endGame();
                    }
                }
            }
        });
    }

    private drawContainers(parent: Drawable, startY: number): number {
        const containerY = startY;
        const containerSpacing = (this.uiManager.width - (RESOURCE_TYPES.length * CONTAINER_SIZE)) / (RESOURCE_TYPES.length + 1);

        this.state.containers.forEach((container, index) => {
            const x = containerSpacing * (index + 1) + index * CONTAINER_SIZE;

            const containerDrawable = parent.addChild(new Drawable({
                x,
                y: containerY,
                width: CONTAINER_SIZE + "px",
                height: CONTAINER_SIZE + "px",
                image: new TextureInfo(80, 80, `minigame/sorter-container-${container.type}`),
                children: [
                    new Drawable({
                        x: 10,
                        y: 10,
                        width: "60px",
                        height: "20px",
                        text: `${container.resources}/${container.maxResources}`,
                        rightAlign: true
                    })
                ]
            }));

            container.x = x;
        });

        return containerY + CONTAINER_SIZE + 20;
    }

    private drawTimer(parent: Drawable, startY: number): number {
        parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: startY,
            width: "200px",
            height: "30px",
            fallbackColor: '#666666',
            image: new TextureInfo(200, 30, "ui/progressbg"),
            children: [
                new Drawable({
                    clipWidth: 0.03 + (this.state.timer / GAME_DURATION) * 0.94,
                    width: "100%",
                    height: "100%",
                    noXStretch: false,
                    fallbackColor: this.state.timer < 10 ? '#ff0000' : '#00ff00',
                    image: new TextureInfo(200, 30, "ui/progressfg")
                })
            ]
        }));

        return startY + 40;
    }

    private drawScore(parent: Drawable, startY: number): number {
        parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: startY,
            width: "200px",
            height: "40px",
            text: `Score: ${this.state.score}`
        }));

        return startY + 50;
    }

    private drawNextResource(parent: Drawable, startY: number): number {
        if (!this.state.currentResource) return startY;

        parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: startY,
            width: "100px",
            height: "100px",
            image: new TextureInfo(100, 100, `minigame/sorter-resource-${this.state.currentResource}`),
            children: [
                new Drawable({
                    anchors: ['centerX'],
                    centerOnOwnX: true,
                    y: 110,
                    width: "100%",
                    text: "Next:"
                })
            ]
        }));

        return startY + 150;
    }

    public onResize(): void {
        this.scroller.onResize();
    }

    public show(): void {
        this.shown = true;
        this.uiManager.frameRequested = true;
    }

    public hide(): void {
        this.shown = false;
    }

    public getLastDrawable(): Drawable | null {
        return this.lastDrawable;
    }

    private toggleRules(): void {
        // Implementation for rules display
        console.log("Toggle rules");
    }
}