// EcoBalance.ts
import { City } from "../game/City.js";
import { UIManager } from "../ui/UIManager.js";
import { Drawable } from "../ui/Drawable.js";
import { TextureInfo } from "../ui/TextureInfo.js";
import { IHasDrawable } from "../ui/IHasDrawable.js";
import { IOnResizeEvent } from "../ui/IOnResizeEvent.js";
import { Resource } from "../game/Resource.js";
import { Flunds } from "../game/ResourceTypes.js";
import { StandardScroller } from "../ui/StandardScroller.js";
import { OnePracticeRun, rangeMapLinear } from "./MinigameUtil.js";

// Game constants
const GAME_DURATION = 90; // seconds
const ZONE_TYPES = ['residential', 'industrial', 'green'] as const;
const RESOURCE_TYPES = ['water', 'air', 'energy'] as const;
const TILE_SIZE = 64;
const ZONE_WIDTH = 3;
const ZONE_HEIGHT = 3;

interface Zone {
    type: typeof ZONE_TYPES[number];
    demand: Record<typeof RESOURCE_TYPES[number], number>;
    current: Record<typeof RESOURCE_TYPES[number], number>;
    capacity: Record<typeof RESOURCE_TYPES[number], number>;
}

interface EcoBalanceState {
    zones: Zone[];
    timer: number;
    score: number;
    connections: [number, number][]; // Pairs of connected zones
    locked: boolean;
}

export class EcoBalance implements IHasDrawable, IOnResizeEvent {
    private state: EcoBalanceState;
    private uiManager: UIManager;
    private scroller: StandardScroller;
    private preloaded = false;
    private shown = false;
    private lastDrawable: Drawable | null = null;
    private gameStarted = false;
    private isPractice = false;
    private timerTimeout: NodeJS.Timeout | null = null;
    private winnings: Resource[] = [];
    private howToPlayShown = false;
    private endReason = "";

    constructor(private city: City, uiManager: UIManager) {
        this.uiManager = uiManager;
        this.scroller = new StandardScroller(false, true);
    }

    public async preloadImages(): Promise<void> {
        if (this.preloaded) return;

        const urls: { [key: string]: string } = {
            "minigame/zone_residential": "assets/minigame/zone_residential.png",
            "minigame/zone_industrial": "assets/minigame/zone_industrial.png",
            "minigame/zone_green": "assets/minigame/zone_green.png",
            "minigame/pipe": "assets/minigame/pipe.png",
            "minigame/pipe_highlight": "assets/minigame/pipe_highlight.png",
            "minigame/water_icon": "assets/minigame/water_icon.png",
            "minigame/air_icon": "assets/minigame/air_icon.png",
            "minigame/energy_icon": "assets/minigame/energy_icon.png",
        };

        await this.uiManager.renderer.loadMoreSprites(this.city, urls);
        this.preloaded = true;
    }

    public show(): void {
        this.shown = true;
        this.uiManager.frameRequested = true;
    }

    public hide(): void {
        this.shown = false;
        this.uiManager.frameRequested = true;
    }

    public startGame(): void {
        if (this.city.checkAndSpendResources(this.getCosts())) {
            this.initializeGame();
            this.city.updateLastUserActionTime();
            this.game.fullSave();
        }
    }

    private initializeGame(): void {
        this.state = {
            zones: ZONE_TYPES.map(type => ({
                type,
                demand: {
                    water: 0,
                    air: 0,
                    energy: 0
                },
                current: {
                    water: 0,
                    air: 0,
                    energy: 0
                },
                capacity: {
                    water: 0,
                    air: 0,
                    energy: 0
                }
            })),
            timer: GAME_DURATION,
            score: 0,
            connections: [],
            locked: false
        };

        this.initializeDemands();
        this.gameStarted = true;
        this.startTimer();
        this.uiManager.frameRequested = true;
    }

    private initializeDemands(): void {
        // Set initial demands based on difficulty
        const difficulty = this.city.minigameOptions.get("eb-d") || "medium";
        const difficultyFactors: Record<string, number> = {
            easy: 0.7,
            medium: 1.0,
            hard: 1.3
        };

        this.state.zones.forEach(zone => {
            switch (zone.type) {
                case 'residential':
                    zone.demand = {
                        water: 50 * difficultyFactors[difficulty],
                        air: 40 * difficultyFactors[difficulty],
                        energy: 30 * difficultyFactors[difficulty]
                    };
                    break;
                case 'industrial':
                    zone.demand = {
                        water: 30 * difficultyFactors[difficulty],
                        air: 20 * difficultyFactors[difficulty],
                        energy: 80 * difficultyFactors[difficulty]
                    };
                    break;
                case 'green':
                    zone.demand = {
                        water: 70 * difficultyFactors[difficulty],
                        air: 60 * difficultyFactors[difficulty],
                        energy: 10 * difficultyFactors[difficulty]
                    };
                    break;
            }

            // Set capacities
            zone.capacity = {
                water: zone.demand.water * 1.2,
                air: zone.demand.air * 1.2,
                energy: zone.demand.energy * 1.2
            };
        });
    }

    private startTimer(): void {
        this.timerTimeout = setTimeout(() => {
            if (!this.gameStarted) return;

            this.state.timer--;
            if (this.state.timer <= 0) {
                this.endReason = "Time's up!";
                this.endGame();
            } else {
                this.startTimer();
            }
            this.uiManager.frameRequested = true;
        }, 1000);
    }

    private connectZones(fromIndex: number, toIndex: number): void {
        if (this.state.locked) return;

        // Check if connection already exists
        const connectionExists = this.state.connections.some(
            ([a, b]) => (a === fromIndex && b === toIndex) || (a === toIndex && b === fromIndex)
        );

        if (connectionExists) return;

        this.state.connections.push([fromIndex, toIndex]);
        this.balanceResources();
    }

    private balanceResources(): void {
        // Simulate resource flow between connected zones
        this.state.zones.forEach(zone => {
            // Reset current to demand
            Object.keys(zone.demand).forEach(resource => {
                zone.current[resource as typeof RESOURCE_TYPES[number]] = zone.demand[resource as typeof RESOURCE_TYPES[number]];
            });

            // Simulate flow from connected zones
            this.state.connections.forEach(([from, to]) => {
                if (to === this.state.zones.indexOf(zone)) {
                    const fromZone = this.state.zones[from];
                    // Transfer 10% of available resources from connected zone
                    Object.keys(zone.demand).forEach(resource => {
                        const resourceKey = resource as typeof RESOURCE_TYPES[number];
                        const available = Math.max(0,
                            fromZone.capacity[resourceKey] -
                            fromZone.demand[resourceKey]);
                        const transfer = Math.floor(available * 0.1);
                        if (transfer > 0) {
                            zone.current[resourceKey] += transfer;
                            fromZone.current[resourceKey] -= transfer;
                        }
                    });
                }
            });

            // Cap at capacity
            Object.keys(zone.capacity).forEach(resource => {
                const resourceKey = resource as typeof RESOURCE_TYPES[number];
                zone.current[resourceKey] = Math.min(
                    zone.current[resourceKey],
                    zone.capacity[resourceKey]
                );
            });
        });

        this.calculateScore();
        this.uiManager.frameRequested = true;
    }

    private calculateScore(): void {
        let score = 0;
        this.state.zones.forEach(zone => {
            Object.keys(zone.demand).forEach(resource => {
                const resourceKey = resource as typeof RESOURCE_TYPES[number];
                const fulfillment = zone.current[resourceKey] / zone.demand[resourceKey];
                score += Math.min(fulfillment, 1) * 100;
            });
        });

        this.state.score = Math.floor(score / 3); // Average across all demands
    }

    private endGame(): void {
        if (!this.gameStarted) return;

        clearTimeout(this.timerTimeout);
        this.gameStarted = false;
        this.state.locked = true;

        this.calculateWinnings();
        this.city.transferResourcesFrom(this.winnings, "earn");
    }

    private calculateWinnings(): void {
        this.winnings = [];

        if (this.isPractice) return;

        const baseReward = Math.floor(rangeMapLinear(
            this.state.score,
            0, 1000,
            0, 3000,
            0.1
        ));

        // Add resource rewards based on performance
        const waterBonus = Math.floor(rangeMapLinear(
            this.state.zones[0].current.water,
            0, this.state.zones[0].capacity.water,
            0, 100,
            0.1
        ));

        const airBonus = Math.floor(rangeMapLinear(
            this.state.zones[1].current.air,
            0, this.state.zones[1].capacity.air,
            0, 100,
            0.1
        ));

        const energyBonus = Math.floor(rangeMapLinear(
            this.state.zones[2].current.energy,
            0, this.state.zones[2].capacity.energy,
            0, 100,
            0.1
        ));

        // Add research points for optimal solutions
        if (this.state.score > 2500) {
            this.winnings.push(new Flunds(baseReward));
            this.city.minigameOptions.set("eb-r", "1"); // Unlock research bonus
        } else {
            this.winnings.push(new Flunds(baseReward));
        }

        // Add resource-specific rewards
        if (waterBonus > 0) this.winnings.push(new Water(waterBonus));
        if (airBonus > 0) this.winnings.push(new Air(airBonus));
        if (energyBonus > 0) this.winnings.push(new Energy(energyBonus));
    }

    public asDrawable(): Drawable {
        if (!this.shown) return this.lastDrawable = new Drawable({ width: "0px" });

        const mainDrawable = new Drawable({
            width: "100%",
            height: "100%",
            fallbackColor: '#222222',
        });

        if (!this.gameStarted) {
            this.drawStartOverlay(mainDrawable);
        } else {
            this.drawGameArea(mainDrawable);
        }

        this.lastDrawable = mainDrawable;
        return mainDrawable;
    }

    private drawStartOverlay(parent: Drawable): void {
        const overlay = parent.addChild(new Drawable({
            anchors: ["centerX"],
            centerOnOwnX: true,
            width: "min(100%, 600px)",
            height: "100%",
            fallbackColor: '#111111',
            onDrag: (x: number, y: number) => { this.scroller.handleDrag(y, overlay.screenArea); },
            onDragEnd: () => { this.scroller.resetDrag(); },
        }));

        if (this.howToPlayShown) {
            this.drawHowToPlay(overlay);
            return;
        }

        let nextY = 10 - this.scroller.getScroll();
        let baseY = nextY;

        overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "100%",
            height: "48px",
            text: "EcoBalance",
        }));
        nextY += 60;

        const startButton = overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "220px",
            height: "48px",
            fallbackColor: '#444444',
            onClick: () => this.startGame(),
            children: [
                new Drawable({
                    anchors: ["centerX"],
                    y: 5,
                    width: "calc(100% - 10px)",
                    height: "100%",
                    text: "Start Game",
                    centerOnOwnX: true
                })
            ]
        }));

        const unaffordable = !this.city.hasResources(this.getCosts(), false);
        addResourceCosts(startButton, this.getCosts(), 86, 58, false, false, false, 48, 10, 32, undefined, undefined, unaffordable, this.city);
        nextY += 60;

        overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "500px",
            height: "48px",
            fallbackColor: '#00000000',
            onClick: () => { this.isPractice = !this.isPractice; },
            children: [
                new Drawable({
                    x: 5,
                    width: "48px",
                    height: "48px",
                    image: new TextureInfo(64, 64, this.isPractice ? "ui/checked" : "ui/unchecked"),
                }),
                new Drawable({
                    anchors: ["right"],
                    rightAlign: true,
                    x: 5,
                    y: 7,
                    width: "calc(100% - 60px)",
                    height: "100%",
                    text: "Practice Run (no rewards)",
                }),
            ]
        }));
        nextY += 60;

        const difficultyOptions = overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "500px",
            height: "48px",
            fallbackColor: '#00000000',
        }));

        ['easy', 'medium', 'hard'].forEach((difficulty, index) => {
            const isSelected = this.city.minigameOptions.get("eb-d") === difficulty;
            difficultyOptions.addChild(new Drawable({
                x: index * 150,
                width: "140px",
                height: "48px",
                fallbackColor: isSelected ? '#555555' : '#333333',
                onClick: () => {
                    this.city.minigameOptions.set("eb-d", difficulty);
                    this.uiManager.frameRequested = true;
                },
                children: [
                    new Drawable({
                        anchors: ["centerX"],
                        centerOnOwnX: true,
                        y: 7,
                        width: "calc(100% - 10px)",
                        height: "100%",
                        text: difficulty.charAt(0).toUpperCase() + difficulty.slice(1),
                        centerOnOwnX: true
                    })
                ]
            }));
        });
        nextY += 60;

        overlay.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: nextY,
            width: "220px",
            height: "48px",
            fallbackColor: '#444444',
            onClick: () => { this.toggleRules(); },
            children: [
                new Drawable({
                    anchors: ["centerX"],
                    y: 5,
                    width: "calc(100% - 10px)",
                    height: "100%",
                    text: "How to Play",
                    centerOnOwnX: true
                })
            ]
        }));
        nextY += 60;

        this.scroller.setChildrenSize(nextY - baseY);
    }

    private drawHowToPlay(parent: Drawable): void {
        parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: 10 - this.scroller.getScroll(),
            width: "100%",
            height: "48px",
            text: "EcoBalance Rules",
        }));

        let nextY = 80 - this.scroller.getScroll();
        const instructions = [
            "Balance resources between three zones to meet their demands:",
            " Residential: Needs clean water, fresh air, and moderate energy",
            " Industrial: Needs water, but produces pollution and requires high energy",
            " Green: Needs water and produces clean air and energy",
            "",
            "Connect zones by tapping the pipes between them to create resource flows.",
            "Each connection allows 10% transfer of available resources.",
            "",
            "Goal: Maintain all zones at or above 80% of their demand for maximum points.",
            "Time: 90 seconds to achieve optimal balance."
        ];

        instructions.forEach(text => {
            parent.addChild(new Drawable({
                x: 20,
                y: nextY,
                width: "calc(100% - 40px)",
                height: "40px",
                wordWrap: true,
                keepParentWidth: true,
                text: text,
            }));
            nextY += 50;
        });

        this.scroller.setChildrenSize(nextY);
    }

    private drawGameArea(parent: Drawable): void {
        // Draw zones
        this.state.zones.forEach((zone, index) => {
            const x = index * (ZONE_WIDTH * TILE_SIZE + 20);
            const zoneDrawable = parent.addChild(new Drawable({
                x: x,
                y: 20,
                width: ZONE_WIDTH * TILE_SIZE + "px",
                height: ZONE_HEIGHT * TILE_SIZE + "px",
                fallbackColor: '#333333',
                image: new TextureInfo(ZONE_WIDTH * TILE_SIZE, ZONE_HEIGHT * TILE_SIZE, `minigame/zone_${zone.type}`),
            }));

            // Draw demand indicators
            this.drawResourceIndicators(zoneDrawable, zone, 20, ZONE_HEIGHT * TILE_SIZE + 40);
        });

        // Draw connections/paths between zones
        for (let i = 0; i < this.state.zones.length; i++) {
            for (let j = i + 1; j < this.state.zones.length; j++) {
                const connectionExists = this.state.connections.some(
                    ([a, b]) => (a === i && b === j) || (a === j && b === i)
                );

                const centerX1 = i * (ZONE_WIDTH * TILE_SIZE + 20) + ZONE_WIDTH * TILE_SIZE / 2;
                const centerX2 = j * (ZONE_WIDTH * TILE_SIZE + 20) + ZONE_WIDTH * TILE_SIZE / 2;
                const centerY = ZONE_HEIGHT * TILE_SIZE / 2 + 20;

                const pipeDrawable = parent.addChild(new Drawable({
                    x: Math.min(centerX1, centerX2),
                    y: centerY - 10,
                    width: Math.abs(centerX2 - centerX1) + "px",
                    height: "20px",
                    image: new TextureInfo(10, 20, connectionExists ? "minigame/pipe_highlight" : "minigame/pipe"),
                    onClick: () => this.connectZones(i, j),
                }));
            }
        }

        // Draw timer
        this.drawTimer(parent);

        // Draw score
        parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: 10,
            width: "200px",
            height: "30px",
            text: `Score: ${this.state.score}`,
            rightAlign: true,
        }));
    }

    private drawResourceIndicators(parent: Drawable, zone: Zone, x: number, y: number): void {
        const resources = ['water', 'air', 'energy'] as const;
        const icons = ['water_icon', 'air_icon', 'energy_icon'] as const;

        resources.forEach((resource, index) => {
            const iconY = y + index * 40;
            const barY = iconY + 30;

            // Icon
            parent.addChild(new Drawable({
                x: x,
                y: iconY,
                width: "32px",
                height: "32px",
                image: new TextureInfo(32, 32, `minigame/${icons[index]}`),
            }));

            // Demand bar background
            parent.addChild(new Drawable({
                x: x + 40,
                y: barY,
                width: "200px",
                height: "20px",
                fallbackColor: '#555555',
                text: `${zone.demand[resource]}`,
                rightAlign: true,
            }));

            // Current fulfillment bar
            const fulfillment = zone.current[resource] / zone.demand[resource];
            parent.addChild(new Drawable({
                x: x + 40,
                y: barY,
                width: `${fulfillment * 200}px`,
                height: "20px",
                fallbackColor: fulfillment > 0.8 ? '#00ff00' :
                    fulfillment > 0.5 ? '#ffff00' : '#ff0000',
            }));
        });
    }

    private drawTimer(parent: Drawable): void {
        parent.addChild(new Drawable({
            anchors: ['centerX'],
            centerOnOwnX: true,
            y: 50,
            width: "200px",
            height: "30px",
            text: `Time: ${Math.floor(this.state.timer / 60)}:${this.state.timer % 60 < 10 ? '0' : ''}${this.state.timer % 60}`,
            rightAlign: true,
        }));
    }

    public getCosts(): { type: string, amount: number }[] {
        return this.isPractice ? OnePracticeRun :
            [{ type: "flunds", amount: 500 }];
    }

    public onResize(): void {
        this.scroller.onResize();
    }

    private toggleRules(): void {
        this.howToPlayShown = !this.howToPlayShown;
        if (this.howToPlayShown) {
            this.scroller.resetScroll();
        }
    }

    getLastDrawable(): Drawable | null {
        return this.lastDrawable;
    }
}
